#include "LVGL_Example.h"

/**********************
 *      TYPEDEFS
 **********************/
typedef enum {
    DISP_SMALL,
    DISP_MEDIUM,
    DISP_LARGE,
} disp_size_t;

/**********************
 *  STATIC PROTOTYPES
 **********************/
static void Onboard_create(lv_obj_t * parent);
static void color_changer_create(lv_obj_t * parent);

static void color_changer_event_cb(lv_event_t * e);
static void color_event_cb(lv_event_t * e);
static void ta_event_cb(lv_event_t * e);
static void birthday_event_cb(lv_event_t * e);
static void calendar_event_cb(lv_event_t * e);
void IRAM_ATTR example1_increase_lvgl_tick(lv_timer_t * t);
static void flow_led_timer_cb(lv_timer_t * t);
/**********************
 *  STATIC VARIABLES
 **********************/
static disp_size_t disp_size;

static lv_obj_t * tv;
static lv_obj_t * calendar;
lv_style_t style_text_muted;
lv_style_t style_title;
static lv_style_t style_icon;
static lv_style_t style_bullet;

static lv_obj_t * chart1;
static lv_obj_t * chart2;
static lv_obj_t * chart3;

static lv_chart_series_t * ser1;
static lv_chart_series_t * ser2;
static lv_chart_series_t * ser3;
static lv_chart_series_t * ser4;

static const lv_font_t * font_large;
static const lv_font_t * font_normal;

static lv_timer_t * auto_step_timer;
static lv_color_t original_screen_bg_color;

static lv_timer_t * meter2_timer;

lv_obj_t * SD_Size;
lv_obj_t * FlashSize;
lv_obj_t * Board_angle;
lv_obj_t * RTC_Time;
lv_obj_t * Wireless_Scan;

CRGB ws2812b_leds[WS2812B_NUM_LEDS];

static const uint8_t flow_led_pins[FLOW_LED_NUM] = FLOW_LED_PINS;
static lv_timer_t * flow_led_timer = NULL;
static bool flow_led_running = false;
static uint8_t flow_led_current = 0;
static uint32_t flow_led_counter = 0;



void IRAM_ATTR auto_switch(lv_timer_t * t)
{
  uint16_t page = lv_tabview_get_tab_act(tv);

  if (page == 0) { 
    lv_tabview_set_act(tv, 1, LV_ANIM_ON); 
  } else if (page == 3) {
    lv_tabview_set_act(tv, 2, LV_ANIM_ON); 
  }
}
void WS2812B_Init(void)
{
  FastLED.addLeds<WS2812B, WS2812B_PIN, GRB>(ws2812b_leds, WS2812B_NUM_LEDS);
  FastLED.setBrightness(50);
  WS2812B_SetColor(0, 0, 0); // 初始化为关闭
  FastLED.show();
}

void WS2812B_SetColor(uint8_t r, uint8_t g, uint8_t b)
{
  for(int i = 0; i < WS2812B_NUM_LEDS; i++) {
    ws2812b_leds[i] = CRGB(r, g, b);
  }
  FastLED.show();
}

void WS2812B_StartupSequence(void)
{
  // 保存原始屏幕背景颜色
  lv_color_t original_bg = lv_obj_get_style_bg_color(lv_scr_act(), 0);
  
  // 红色页面 + 红灯
  lv_obj_set_style_bg_color(lv_scr_act(), lv_color_hex(0xFF0000), 0);
  lv_obj_set_style_bg_opa(lv_scr_act(), LV_OPA_COVER, 0);
  WS2812B_SetColor(255, 0, 0); // 红色
  lv_refr_now(NULL);
  delay(500);
  
  // 绿色页面 + 绿灯
  lv_obj_set_style_bg_color(lv_scr_act(), lv_color_hex(0x00FF00), 0);
  lv_obj_set_style_bg_opa(lv_scr_act(), LV_OPA_COVER, 0);
  WS2812B_SetColor(0, 255, 0); // 绿色
  lv_refr_now(NULL);
  delay(500);
  
  // 蓝色页面 + 蓝灯
  lv_obj_set_style_bg_color(lv_scr_act(), lv_color_hex(0x0000FF), 0);
  lv_obj_set_style_bg_opa(lv_scr_act(), LV_OPA_COVER, 0);
  WS2812B_SetColor(0, 0, 255); // 蓝色
  lv_refr_now(NULL);
  delay(500);
  
  // 恢复原始背景颜色
  lv_obj_set_style_bg_color(lv_scr_act(), original_bg, 0);
  WS2812B_SetColor(0, 0, 0); // 关闭LED
  lv_refr_now(NULL);
}

void FlowLED_Init(void)
{
  // 初始化流水灯引脚为输出
  for(int i = 0; i < FLOW_LED_NUM; i++) {
    pinMode(flow_led_pins[i], OUTPUT);
    digitalWrite(flow_led_pins[i], LOW); // 初始状态为关闭
  }
  
  // 初始化按键引脚为输入，使用内部上拉电阻
  // 如果按键按下时引脚被拉低到GND，则读取值为LOW
  // 如果按键按下时引脚被拉高到VCC，则读取值为HIGH
  pinMode(BUTTON_PIN, INPUT_PULLUP);
}

void FlowLED_Test(void)
{
  if(flow_led_running) {
    return; // 如果已经在运行，则不重复启动
  }
  
  flow_led_running = true;
  flow_led_current = 0;
  flow_led_counter = 0;
  
  // 关闭所有LED
  for(int i = 0; i < FLOW_LED_NUM; i++) {
    digitalWrite(flow_led_pins[i], LOW);
  }
  
  // 立即点亮第一个LED
  digitalWrite(flow_led_pins[0], HIGH);
  
  // 创建定时器，每100ms执行一次（10次 = 1秒）
  if(flow_led_timer == NULL) {
    flow_led_timer = lv_timer_create(flow_led_timer_cb, 100, NULL);
  }
}

static void flow_led_timer_cb(lv_timer_t * t)
{
  if(!flow_led_running) {
    return;
  }
  
  flow_led_counter++;
  
  // 每10次（1秒）切换一次LED
  if(flow_led_counter >= 10) {
    flow_led_counter = 0;
    
    // 关闭当前LED
    digitalWrite(flow_led_pins[flow_led_current], LOW);
    
    // 移动到下一个LED
    flow_led_current++;
    
    // 如果完成一轮，停止流水灯
    if(flow_led_current >= FLOW_LED_NUM) {
      // 停止流水灯
      flow_led_running = false;
      
      // 删除定时器
      if(flow_led_timer != NULL) {
        lv_timer_del(flow_led_timer);
        flow_led_timer = NULL;
      }
      
      return;
    }
    
    // 点亮下一个LED
    digitalWrite(flow_led_pins[flow_led_current], HIGH);
  }
}

void FlowLED_CheckButton(lv_timer_t * t)
{
  static uint32_t last_check_time = 0;
  static bool button_pressed = false;
  static bool last_button_state = HIGH;
  static uint32_t stable_count = 0;
  const uint32_t check_interval = 10; // 每10ms检查一次
  const uint32_t stable_threshold = 3; // 需要连续3次相同状态才认为稳定（30ms防抖）
  
  uint32_t current_time = millis();
  
  // 限制检查频率
  if(current_time - last_check_time < check_interval) {
    return;
  }
  last_check_time = current_time;
  
  bool current_button_state = digitalRead(BUTTON_PIN);
  
  // 简单防抖：如果状态稳定
  if(current_button_state == last_button_state) {
    stable_count++;
  } else {
    stable_count = 0;
  }
  
  // 只有当状态稳定时才处理
  if(stable_count >= stable_threshold) {
    // 检测按键按下边沿（从HIGH变为LOW，使用上拉电阻时按下为LOW）
    if(current_button_state == LOW && last_button_state == HIGH && !button_pressed) {
      button_pressed = true;
      FlowLED_Test(); // 启动流水灯测试
    }
    // 按键释放时重置标志
    else if(current_button_state == HIGH && button_pressed) {
      button_pressed = false;
    }
  }
  
  last_button_state = current_button_state;
}

void Lvgl_Example1(void){
  
  // 初始化WS2812B
  WS2812B_Init();
  
  // 初始化流水灯和按键
  FlowLED_Init();
  
  // 执行启动序列
  WS2812B_StartupSequence();

  // 创建按键检测定时器，每50ms检测一次
  lv_timer_create(FlowLED_CheckButton, 50, NULL);

  disp_size = DISP_SMALL;                            

  font_large = LV_FONT_DEFAULT;                             
  font_normal = LV_FONT_DEFAULT;                         
  
  lv_coord_t tab_h;
  tab_h = 30;
#if LV_FONT_MONTSERRAT_20
    font_large     = &lv_font_montserrat_20;
#elif LV_FONT_MONTSERRAT_18
    font_large     = &lv_font_montserrat_18;
  #else
    LV_LOG_WARN("LV_FONT_MONTSERRAT_18 is not enabled for the widgets demo. Using LV_FONT_DEFAULT instead.");
  #endif
#if LV_FONT_MONTSERRAT_14
    font_normal    = &lv_font_montserrat_14;
#elif LV_FONT_MONTSERRAT_12
    font_normal    = &lv_font_montserrat_12;
  #else
    LV_LOG_WARN("LV_FONT_MONTSERRAT_12 is not enabled for the widgets demo. Using LV_FONT_DEFAULT instead.");
  #endif
  
  lv_style_init(&style_text_muted);
  lv_style_set_text_opa(&style_text_muted, LV_OPA_90);

  lv_style_init(&style_title);
  lv_style_set_text_font(&style_title, font_large);

  lv_style_init(&style_icon);
  lv_style_set_text_color(&style_icon, lv_theme_get_color_primary(NULL));
  lv_style_set_text_font(&style_icon, font_large);

  lv_style_init(&style_bullet);
  lv_style_set_border_width(&style_bullet, 0);
  lv_style_set_radius(&style_bullet, LV_RADIUS_CIRCLE);

  tv = lv_tabview_create(lv_scr_act(), LV_DIR_TOP, tab_h);
  lv_obj_set_style_pad_all(tv, 0, LV_PART_MAIN);
  lv_obj_t * tab_content = lv_tabview_get_content(tv);
  lv_obj_set_style_pad_all(tab_content, 0, LV_PART_MAIN);
  lv_obj_set_style_pad_left(tab_content, 4, LV_PART_MAIN);
  lv_obj_set_style_pad_right(tab_content, 4, LV_PART_MAIN);

  lv_obj_set_style_text_font(lv_scr_act(), font_normal, 0);


  lv_obj_t * t1 = lv_tabview_add_tab(tv, "Onboard");
  
  
  Onboard_create(t1);
  
  
}

void Lvgl_Example1_close(void)
{
  /*Delete all animation*/
  lv_anim_del(NULL, NULL);

  lv_timer_del(meter2_timer);
  meter2_timer = NULL;

  lv_obj_clean(lv_scr_act());

  lv_style_reset(&style_text_muted);
  lv_style_reset(&style_title);
  lv_style_reset(&style_icon);
  lv_style_reset(&style_bullet);
}


/**********************
*   STATIC FUNCTIONS
**********************/

static void Onboard_create(lv_obj_t * parent)
{

  /*Create a panel*/
  lv_obj_t * panel1 = lv_obj_create(parent);
  lv_obj_set_height(panel1, LV_SIZE_CONTENT);
  lv_obj_set_style_pad_left(parent, 2, LV_PART_MAIN);
  lv_obj_set_style_pad_right(parent, 2, LV_PART_MAIN);
  lv_obj_set_style_pad_top(parent, 0, LV_PART_MAIN);
  lv_obj_set_style_pad_bottom(parent, 0, LV_PART_MAIN);
  lv_obj_set_style_pad_left(panel1, 5, LV_PART_MAIN);
  lv_obj_set_style_pad_right(panel1, 5, LV_PART_MAIN);
  lv_obj_set_style_pad_top(panel1, 4, LV_PART_MAIN);
  lv_obj_set_style_pad_bottom(panel1, 4, LV_PART_MAIN);
  lv_obj_set_style_pad_column(panel1, 0, LV_PART_MAIN);
  lv_obj_set_style_pad_row(panel1, 4, LV_PART_MAIN);

  lv_obj_t * panel1_title = lv_label_create(panel1);
  lv_label_set_text(panel1_title, "Onboard parameter");
  lv_obj_add_style(panel1_title, &style_title, 0);

  lv_obj_t * SD_label = lv_label_create(panel1);
  lv_label_set_text(SD_label, "SD Card");
  lv_obj_add_style(SD_label, &style_text_muted, 0);

  SD_Size = lv_textarea_create(panel1);
  lv_textarea_set_one_line(SD_Size, true);
  lv_textarea_set_placeholder_text(SD_Size, "SD Size");
  lv_obj_add_event_cb(SD_Size, ta_event_cb, LV_EVENT_ALL, NULL);

  lv_obj_t * Flash_label = lv_label_create(panel1);
  lv_label_set_text(Flash_label, "Flash Size");
  lv_obj_add_style(Flash_label, &style_text_muted, 0);

  FlashSize = lv_textarea_create(panel1);
  lv_textarea_set_one_line(FlashSize, true);
  lv_textarea_set_placeholder_text(FlashSize, "Flash Size");
  lv_obj_add_event_cb(FlashSize, ta_event_cb, LV_EVENT_ALL, NULL);

  lv_obj_t * Wireless_label = lv_label_create(panel1);
  lv_label_set_text(Wireless_label, "Wireless scan");
  lv_obj_add_style(Wireless_label, &style_text_muted, 0);

  Wireless_Scan = lv_textarea_create(panel1);
  lv_textarea_set_one_line(Wireless_Scan, true);
  lv_textarea_set_placeholder_text(Wireless_Scan, "Wireless number");
  lv_obj_add_event_cb(Wireless_Scan, ta_event_cb, LV_EVENT_ALL, NULL);

  // 器件布局
  static lv_coord_t grid_main_col_dsc[] = {LV_GRID_FR(1), LV_GRID_TEMPLATE_LAST};
  static lv_coord_t grid_main_row_dsc[] = {LV_GRID_CONTENT, LV_GRID_TEMPLATE_LAST};
  lv_obj_set_grid_dsc_array(parent, grid_main_col_dsc, grid_main_row_dsc);


  /*Create the top panel*/
  static lv_coord_t grid_1_col_dsc[] = {LV_GRID_CONTENT, LV_GRID_TEMPLATE_LAST};
  static lv_coord_t grid_1_row_dsc[] = {LV_GRID_CONTENT, /*Avatar*/
                                        LV_GRID_CONTENT, /*Name*/
                                        LV_GRID_CONTENT, /*Description*/
                                        LV_GRID_CONTENT, /*Email*/
                                        LV_GRID_CONTENT, /*Phone number*/
                                        LV_GRID_CONTENT, /*Button1*/
                                        LV_GRID_CONTENT, /*Button2*/
                                        LV_GRID_TEMPLATE_LAST
                                        };

  lv_obj_set_grid_dsc_array(panel1, grid_1_col_dsc, grid_1_row_dsc);


  static lv_coord_t grid_2_col_dsc[] = {LV_GRID_FR(1), LV_GRID_TEMPLATE_LAST};
  static lv_coord_t grid_2_row_dsc[] = {
    LV_GRID_CONTENT,  /*Title*/
    0,                /*Separator*/
    LV_GRID_CONTENT,  /*SD label*/
    39,               /*SD box*/
    LV_GRID_CONTENT,  /*Flash label*/
    39,               /*Flash box*/
    LV_GRID_CONTENT,  /*Wireless label*/
    39,               /*Wireless box*/
    LV_GRID_TEMPLATE_LAST
  };

  // lv_obj_set_grid_dsc_array(panel2, grid_2_col_dsc, grid_2_row_dsc);
  // lv_obj_set_grid_dsc_array(panel3, grid_2_col_dsc, grid_2_row_dsc);

  lv_obj_set_grid_cell(panel1, LV_GRID_ALIGN_STRETCH, 0, 1, LV_GRID_ALIGN_STRETCH, 0, 1);
  lv_obj_set_grid_dsc_array(panel1, grid_2_col_dsc, grid_2_row_dsc);
  lv_obj_set_grid_cell(panel1_title, LV_GRID_ALIGN_START, 0, 1, LV_GRID_ALIGN_CENTER, 0, 1);
  lv_obj_set_grid_cell(SD_label, LV_GRID_ALIGN_START, 0, 1, LV_GRID_ALIGN_START, 2, 1);
  lv_obj_set_grid_cell(SD_Size, LV_GRID_ALIGN_STRETCH, 0, 1, LV_GRID_ALIGN_CENTER, 3, 1);
  lv_obj_set_height(SD_Size, 25);
  lv_obj_set_style_pad_top(SD_Size, 1, LV_PART_MAIN);
  lv_obj_set_style_pad_bottom(SD_Size, 3, LV_PART_MAIN);
  lv_obj_set_style_text_font(SD_Size, font_normal, LV_PART_MAIN);
  lv_obj_set_grid_cell(Flash_label, LV_GRID_ALIGN_START, 0, 1, LV_GRID_ALIGN_START, 4, 1);
  lv_obj_set_grid_cell(FlashSize, LV_GRID_ALIGN_STRETCH, 0, 1, LV_GRID_ALIGN_CENTER, 5, 1);
  lv_obj_set_height(FlashSize, 25);
  lv_obj_set_style_pad_top(FlashSize, 1, LV_PART_MAIN);
  lv_obj_set_style_pad_bottom(FlashSize, 3, LV_PART_MAIN);
  lv_obj_set_style_text_font(FlashSize, font_normal, LV_PART_MAIN);
  lv_obj_set_grid_cell(Wireless_label, LV_GRID_ALIGN_START, 0, 1, LV_GRID_ALIGN_START, 6, 1);
  lv_obj_set_grid_cell(Wireless_Scan, LV_GRID_ALIGN_STRETCH, 0, 1, LV_GRID_ALIGN_CENTER, 7, 1);
  lv_obj_set_height(Wireless_Scan,25);
  lv_obj_set_style_pad_top(Wireless_Scan, 1, LV_PART_MAIN);
  lv_obj_set_style_pad_bottom(Wireless_Scan, 3, LV_PART_MAIN);
  lv_obj_set_style_text_font(Wireless_Scan, font_normal, LV_PART_MAIN);

  // 器件布局 END
  
  auto_step_timer = lv_timer_create(example1_increase_lvgl_tick, 100, NULL);
}

void IRAM_ATTR example1_increase_lvgl_tick(lv_timer_t * t)
{
  char buf[100]={0}; 
  
  snprintf(buf, sizeof(buf), "%d MB\r\n", SDCard_Size);
  lv_textarea_set_placeholder_text(SD_Size, buf);
  snprintf(buf, sizeof(buf), "%d MB\r\n", Flash_Size);
  lv_textarea_set_placeholder_text(FlashSize, buf);
  if(Scan_finish)
    snprintf(buf, sizeof(buf), "W: %d  B: %d    OK.\r\n",WIFI_NUM,BLE_NUM);
    // snprintf(buf, sizeof(buf), "WIFI: %d     ..OK.\r\n",WIFI_NUM);
  else
    snprintf(buf, sizeof(buf), "W: %d  B: %d\r\n",WIFI_NUM,BLE_NUM);
    // snprintf(buf, sizeof(buf), "WIFI: %d  \r\n",WIFI_NUM);
  lv_textarea_set_placeholder_text(Wireless_Scan, buf);
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
static void color_changer_create(lv_obj_t * parent)
{
    static lv_palette_t palette[] = {
        LV_PALETTE_BLUE, LV_PALETTE_GREEN, LV_PALETTE_BLUE_GREY,  LV_PALETTE_ORANGE,
        LV_PALETTE_RED, LV_PALETTE_PURPLE, LV_PALETTE_TEAL, _LV_PALETTE_LAST
    };

    lv_obj_t * color_cont = lv_obj_create(parent);
    lv_obj_remove_style_all(color_cont);
    lv_obj_set_flex_flow(color_cont, LV_FLEX_FLOW_ROW);
    lv_obj_set_flex_align(color_cont, LV_FLEX_ALIGN_SPACE_EVENLY, LV_FLEX_ALIGN_CENTER, LV_FLEX_ALIGN_CENTER);
    lv_obj_add_flag(color_cont, LV_OBJ_FLAG_FLOATING);

    lv_obj_set_style_bg_color(color_cont, lv_color_white(), 0);
    lv_obj_set_style_pad_right(color_cont, disp_size == DISP_SMALL ? LV_DPX(47) : LV_DPX(55), 0);
    lv_obj_set_style_bg_opa(color_cont, LV_OPA_COVER, 0);
    lv_obj_set_style_radius(color_cont, LV_RADIUS_CIRCLE, 0);

    if(disp_size == DISP_SMALL) lv_obj_set_size(color_cont, LV_DPX(52), LV_DPX(52));
    else lv_obj_set_size(color_cont, LV_DPX(60), LV_DPX(60));

    lv_obj_align(color_cont, LV_ALIGN_BOTTOM_RIGHT, - LV_DPX(10),  - LV_DPX(10));

    uint32_t i;
    for(i = 0; palette[i] != _LV_PALETTE_LAST; i++) {
        lv_obj_t * c = lv_btn_create(color_cont);
        lv_obj_set_style_bg_color(c, lv_palette_main(palette[i]), 0);
        lv_obj_set_style_radius(c, LV_RADIUS_CIRCLE, 0);
        lv_obj_set_style_opa(c, LV_OPA_TRANSP, 0);
        lv_obj_set_size(c, 20, 20);
        lv_obj_add_event_cb(c, color_event_cb, LV_EVENT_ALL, &palette[i]);
        lv_obj_clear_flag(c, LV_OBJ_FLAG_SCROLL_ON_FOCUS);
    }

    lv_obj_t * btn = lv_btn_create(parent);
    lv_obj_add_flag(btn, LV_OBJ_FLAG_FLOATING | LV_OBJ_FLAG_CLICKABLE);
    lv_obj_set_style_bg_color(btn, lv_color_white(), LV_STATE_CHECKED);
    lv_obj_set_style_pad_all(btn, 10, 0);
    lv_obj_set_style_radius(btn, LV_RADIUS_CIRCLE, 0);
    lv_obj_add_event_cb(btn, color_changer_event_cb, LV_EVENT_ALL, color_cont);
    lv_obj_set_style_shadow_width(btn, 0, 0);
    lv_obj_set_style_bg_img_src(btn, LV_SYMBOL_TINT, 0);

    if(disp_size == DISP_SMALL) {
        lv_obj_set_size(btn, LV_DPX(42), LV_DPX(42));
        lv_obj_align(btn, LV_ALIGN_BOTTOM_RIGHT, -LV_DPX(15), -LV_DPX(15));
    }
    else {
        lv_obj_set_size(btn, LV_DPX(50), LV_DPX(50));
        lv_obj_align(btn, LV_ALIGN_BOTTOM_RIGHT, -LV_DPX(15), -LV_DPX(15));
    }
}

static void color_changer_anim_cb(void * var, int32_t v)
{
    lv_obj_t * obj =(lv_obj_t *) var;
    lv_coord_t max_w = lv_obj_get_width(lv_obj_get_parent(obj)) - LV_DPX(20);
    lv_coord_t w;

    if(disp_size == DISP_SMALL) {
        w = lv_map(v, 0, 256, LV_DPX(52), max_w);
        lv_obj_set_width(obj, w);
        lv_obj_align(obj, LV_ALIGN_BOTTOM_RIGHT, - LV_DPX(10),  - LV_DPX(10));
    }
    else {
        w = lv_map(v, 0, 256, LV_DPX(60), max_w);
        lv_obj_set_width(obj, w);
        lv_obj_align(obj, LV_ALIGN_BOTTOM_RIGHT, - LV_DPX(10),  - LV_DPX(10));
    }

    if(v > LV_OPA_COVER) v = LV_OPA_COVER;

    uint32_t i;
    for(i = 0; i < lv_obj_get_child_cnt(obj); i++) {
        lv_obj_set_style_opa(lv_obj_get_child(obj, i), v, 0);
    }

}

static void color_changer_event_cb(lv_event_t * e)
{
    if(lv_event_get_code(e) == LV_EVENT_CLICKED) {
        lv_obj_t * color_cont = (lv_obj_t *)lv_event_get_user_data(e);
        if(lv_obj_get_width(color_cont) < LV_HOR_RES / 2) {
            lv_anim_t a;
            lv_anim_init(&a);
            lv_anim_set_var(&a, color_cont);
            lv_anim_set_exec_cb(&a, color_changer_anim_cb);
            lv_anim_set_values(&a, 0, 256);
            lv_anim_set_time(&a, 200);
            lv_anim_start(&a);
        }
        else {
            lv_anim_t a;
            lv_anim_init(&a);
            lv_anim_set_var(&a, color_cont);
            lv_anim_set_exec_cb(&a, color_changer_anim_cb);
            lv_anim_set_values(&a, 256, 0);
            lv_anim_set_time(&a, 200);
            lv_anim_start(&a);
        }
    }
}
static void color_event_cb(lv_event_t * e)
{
    lv_event_code_t code = lv_event_get_code(e);
    lv_obj_t * obj = lv_event_get_target(e);

    if(code == LV_EVENT_FOCUSED) {
        lv_obj_t * color_cont = lv_obj_get_parent(obj);
        if(lv_obj_get_width(color_cont) < LV_HOR_RES / 2) {
            lv_anim_t a;
            lv_anim_init(&a);
            lv_anim_set_var(&a, color_cont);
            lv_anim_set_exec_cb(&a, color_changer_anim_cb);
            lv_anim_set_values(&a, 0, 256);
            lv_anim_set_time(&a, 200);
            lv_anim_start(&a);
        }
    }
    else if(code == LV_EVENT_CLICKED) {
        lv_palette_t * palette_primary = (lv_palette_t *)lv_event_get_user_data(e);
        lv_palette_t palette_secondary =(lv_palette_t) ((*palette_primary) + 3); /*Use another palette as secondary*/
        if(palette_secondary >= _LV_PALETTE_LAST) palette_secondary =(lv_palette_t)(0);
#if LV_USE_THEME_DEFAULT
        lv_theme_default_init(NULL, lv_palette_main(*palette_primary), lv_palette_main(palette_secondary),
                              LV_THEME_DEFAULT_DARK, font_normal);
#endif
        lv_color_t color = lv_palette_main(*palette_primary);
        lv_style_set_text_color(&style_icon, color);
        lv_chart_set_series_color(chart1, ser1, color);
        lv_chart_set_series_color(chart2, ser3, color);
    }
}



static void ta_event_cb(lv_event_t * e)
{
}

static void birthday_event_cb(lv_event_t * e)
{
    lv_event_code_t code = lv_event_get_code(e);
    lv_obj_t * ta = lv_event_get_target(e);

    if(code == LV_EVENT_FOCUSED) {
        if(lv_indev_get_type(lv_indev_get_act()) == LV_INDEV_TYPE_POINTER) {
            if(calendar == NULL) {
                lv_obj_add_flag(lv_layer_top(), LV_OBJ_FLAG_CLICKABLE);
                calendar = lv_calendar_create(lv_layer_top());
                lv_obj_set_style_bg_opa(lv_layer_top(), LV_OPA_50, 0);
                lv_obj_set_style_bg_color(lv_layer_top(), lv_palette_main(LV_PALETTE_GREY), 0);
                if(disp_size == DISP_SMALL) lv_obj_set_size(calendar, 180, 200);
                else if(disp_size == DISP_MEDIUM) lv_obj_set_size(calendar, 200, 220);
                else  lv_obj_set_size(calendar, 300, 330);
                lv_calendar_set_showed_date(calendar, 1990, 01);
                lv_obj_align(calendar, LV_ALIGN_CENTER, 0, 30);
                lv_obj_add_event_cb(calendar, calendar_event_cb, LV_EVENT_ALL, ta);

                lv_calendar_header_dropdown_create(calendar);
            }
        }
    }
}


static void calendar_event_cb(lv_event_t * e)
{
    lv_event_code_t code = lv_event_get_code(e);
    lv_obj_t * ta = (lv_obj_t *)lv_event_get_user_data(e);
    lv_obj_t * obj = (lv_obj_t *)lv_event_get_current_target(e);
    if(code == LV_EVENT_VALUE_CHANGED) {
        lv_calendar_date_t d;
        lv_calendar_get_pressed_date(obj, &d);
        char buf[32];
        lv_snprintf(buf, sizeof(buf), "%02d.%02d.%d", d.day, d.month, d.year);
        lv_textarea_set_text(ta, buf);

        lv_obj_del(calendar);
        calendar = NULL;
        lv_obj_clear_flag(lv_layer_top(), LV_OBJ_FLAG_CLICKABLE);
        lv_obj_set_style_bg_opa(lv_layer_top(), LV_OPA_TRANSP, 0);
    }
}



