#include <stdio.h>
#include <string.h>
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "driver/uart.h"
#include "driver/gpio.h"

// UART configuration
#define RX_PIN 17
#define TX_PIN 18
#define BAUD_RATE 115200
#define UART_NUM UART_NUM_1
#define BUF_SIZE 1024
#define RESPONSE_TIMEOUT_MS 2000

// Initialize UART
void init_uart() {
    uart_config_t uart_config = {
        .baud_rate = BAUD_RATE,
        .data_bits = UART_DATA_8_BITS,
        .parity = UART_PARITY_DISABLE,
        .stop_bits = UART_STOP_BITS_1,
        .flow_ctrl = UART_HW_FLOWCTRL_DISABLE,
        .source_clk = UART_SCLK_DEFAULT,
    };
    
    ESP_ERROR_CHECK(uart_driver_install(UART_NUM, BUF_SIZE * 2, 0, 0, NULL, 0));
    ESP_ERROR_CHECK(uart_param_config(UART_NUM, &uart_config));
    ESP_ERROR_CHECK(uart_set_pin(UART_NUM, TX_PIN, RX_PIN, UART_PIN_NO_CHANGE, UART_PIN_NO_CHANGE));
}

// Send command and wait for response
bool send_command(const char* cmd, const char* expected_response, uint32_t timeout_ms) {
    uint8_t data[BUF_SIZE];
    int total_len = 0;
    uint32_t start_time = xTaskGetTickCount() * portTICK_PERIOD_MS;
    
    // Send command
    printf("Send: %s\n", cmd);
    uart_write_bytes(UART_NUM, cmd, strlen(cmd));
    uart_write_bytes(UART_NUM, "\r\n", 2);
    
    // Wait for response
    while ((xTaskGetTickCount() * portTICK_PERIOD_MS - start_time) < timeout_ms) {
        int len = uart_read_bytes(UART_NUM, data + total_len, BUF_SIZE - total_len - 1, 20 / portTICK_PERIOD_MS);
        if (len > 0) {
            total_len += len;
            data[total_len] = '\0';
            
            // Check if expected response is received
            if (expected_response == NULL || strstr((char*)data, expected_response) != NULL) {
                printf("Recv: %s\n", data);
                return true;
            }
        }
    }
    
    printf("Timeout waiting for response (Received: %.*s)\n", total_len, data);
    return false;
}

void app_main() {
    init_uart();
    
    // Wait for module to be ready
    while (!send_command("AT", "OK", RESPONSE_TIMEOUT_MS)) {
        vTaskDelay(1000 / portTICK_PERIOD_MS);
        printf("Retrying AT command...\n");
    }
    
    // Send a series of AT commands
    const char* commands[] = {
        "ATE1",         // Enable echo
        "AT+CPIN?",     // Check SIM card status
        "AT+COPS?",     // Query network operator
        "AT+CGDCONT?",  // Query PDP context
        "AT+CGREG?",    // Query network registration status
        "AT+SIMCOMATI", // Query module information
        "ATD10086;"     // Make a call (China Mobile customer service)
    };
    
    for (int i = 0; i < sizeof(commands)/sizeof(commands[0]); i++) {
        send_command(commands[i], "OK", RESPONSE_TIMEOUT_MS);
        vTaskDelay(500 / portTICK_PERIOD_MS);
    }
    
    // Continuously receive data
    uint8_t data[BUF_SIZE];
    while (1) {
        int len = uart_read_bytes(UART_NUM, data, BUF_SIZE - 1, 100 / portTICK_PERIOD_MS);
        if (len > 0) {
            data[len] = '\0';
            printf("%s", data);
        }
        vTaskDelay(10 / portTICK_PERIOD_MS);
    }
}