/*
 * SPDX-FileCopyrightText: 2024-2026 Espressif Systems (Shanghai) CO LTD
 *
 * SPDX-License-Identifier: Apache-2.0
 */

#include "freertos/FreeRTOS.h"
#include "freertos/semphr.h"
#include "esp_log.h"
#include "unity.h"
#include "unity_test_utils_memory.h"

#include "lvgl.h"
#include "lv_eaf.h"

static const char *TAG = "eaf player";

#define TEST_LCD_H_RES      240
#define TEST_LCD_V_RES      240

/* Embedded EAF file - symbols generated by EMBED_TXTFILES */
extern const uint8_t test_eaf_start[] asm("_binary_test_eaf_start");
extern const uint8_t test_eaf_end[] asm("_binary_test_eaf_end");

static SemaphoreHandle_t flush_sem;
static lv_disp_drv_t *s_disp_drv;
static lv_disp_draw_buf_t *s_disp_buf;
static lv_color_t *s_draw_buf_mem;

static void test_flush_callback(lv_disp_drv_t *drv, const lv_area_t *area, lv_color_t *color_map)
{
    LV_UNUSED(color_map);
    ESP_LOGI(TAG, "flush_cb [%d,%d,%d,%d]", area->x1, area->y1, area->x2, area->y2);
    lv_disp_flush_ready(drv);
    xSemaphoreGive(flush_sem);
}

static void lvgl_test_env_init(void)
{
    lv_init();

    uint32_t buffer_pixels = TEST_LCD_H_RES * TEST_LCD_V_RES;
    s_draw_buf_mem = heap_caps_malloc(buffer_pixels * sizeof(lv_color_t), MALLOC_CAP_DEFAULT);
    TEST_ASSERT_NOT_NULL(s_draw_buf_mem);

    s_disp_buf = heap_caps_malloc(sizeof(lv_disp_draw_buf_t), MALLOC_CAP_DEFAULT);
    TEST_ASSERT_NOT_NULL(s_disp_buf);
    lv_disp_draw_buf_init(s_disp_buf, s_draw_buf_mem, NULL, buffer_pixels);

    s_disp_drv = heap_caps_malloc(sizeof(lv_disp_drv_t), MALLOC_CAP_DEFAULT);
    TEST_ASSERT_NOT_NULL(s_disp_drv);
    lv_disp_drv_init(s_disp_drv);
    s_disp_drv->hor_res = TEST_LCD_H_RES;
    s_disp_drv->ver_res = TEST_LCD_V_RES;
    s_disp_drv->flush_cb = test_flush_callback;
    s_disp_drv->draw_buf = s_disp_buf;
    lv_disp_drv_register(s_disp_drv);

    flush_sem = xSemaphoreCreateBinary();
    TEST_ASSERT_NOT_NULL(flush_sem);
}

static void lvgl_test_env_deinit(void)
{
    free(s_disp_drv);
    free(s_disp_buf);
    s_disp_drv = NULL;
    s_disp_buf = NULL;
    free(s_draw_buf_mem);
    s_draw_buf_mem = NULL;

    lv_deinit();

    vSemaphoreDelete(flush_sem);
    flush_sem = NULL;
}

TEST_CASE("Validate EAF player loads built-in animation", "[eaf][lv_eaf_set_src]")
{
    lvgl_test_env_init();

    size_t eaf_size = test_eaf_end - test_eaf_start;
    ESP_LOGI(TAG, "Embedded EAF size: %d bytes", eaf_size);

    lv_obj_t *anim = lv_eaf_create(lv_scr_act());
    TEST_ASSERT_NOT_NULL(anim);

    lv_eaf_set_src_data(anim, test_eaf_start, eaf_size);
    TEST_ASSERT_TRUE(lv_eaf_is_loaded(anim));

    int32_t total_frames = lv_eaf_get_total_frames(anim);
    ESP_LOGI(TAG, "Total frames: %d", total_frames);
    TEST_ASSERT_GREATER_THAN(0, total_frames);
    TEST_ASSERT_EQUAL_INT32(0, lv_eaf_get_current_frame(anim));

    lv_refr_now(NULL);
    TEST_ASSERT_TRUE(xSemaphoreTake(flush_sem, pdMS_TO_TICKS(3000)));

    lv_obj_del(anim);
    lvgl_test_env_deinit();
}

#define TEST_MEMORY_LEAK_THRESHOLD  (700)

static size_t before_free_8bit;
static size_t before_free_32bit;

void setUp(void)
{
    before_free_8bit = heap_caps_get_free_size(MALLOC_CAP_8BIT);
    before_free_32bit = heap_caps_get_free_size(MALLOC_CAP_32BIT);
}

void tearDown(void)
{
    size_t after_free_8bit = heap_caps_get_free_size(MALLOC_CAP_8BIT);
    size_t after_free_32bit = heap_caps_get_free_size(MALLOC_CAP_32BIT);
    unity_utils_check_leak(before_free_8bit, after_free_8bit, "8BIT", TEST_MEMORY_LEAK_THRESHOLD);
    unity_utils_check_leak(before_free_32bit, after_free_32bit, "32BIT", TEST_MEMORY_LEAK_THRESHOLD);
}

void app_main(void)
{
    unity_run_menu();
}
