/*
 * SPDX-FileCopyrightText: 2024-2026 Espressif Systems (Shanghai) CO LTD
 *
 * SPDX-License-Identifier: Apache-2.0
 */

#include <string.h>
#include "freertos/FreeRTOS.h"
#include "freertos/semphr.h"
#include "esp_heap_caps.h"
#include "unity.h"
#include "unity_test_utils_memory.h"

#include "lvgl.h"
#include "esp_lv_lottie.h"

#define TEST_LCD_H_RES      240
#define TEST_LCD_V_RES      240

/* Embedded Lottie JSON - symbols generated by EMBED_TXTFILES */
extern const uint8_t welcome_json_start[] asm("_binary_welcome_json_start");
extern const uint8_t welcome_json_end[] asm("_binary_welcome_json_end");

static SemaphoreHandle_t s_flush_sem;
static lv_display_t *s_display;
static lv_color_t *s_draw_buf;

static void test_flush_callback(lv_display_t *disp, const lv_area_t *area, uint8_t *px_map)
{
    LV_UNUSED(area);
    LV_UNUSED(px_map);
    lv_display_flush_ready(disp);
    xSemaphoreGive(s_flush_sem);
}

static void lvgl_test_env_init(void)
{
    lv_init();

    uint32_t buffer_pixels = TEST_LCD_H_RES * TEST_LCD_V_RES;
    s_draw_buf = heap_caps_malloc(buffer_pixels * sizeof(lv_color_t), MALLOC_CAP_DEFAULT);
    TEST_ASSERT_NOT_NULL(s_draw_buf);

    s_display = lv_display_create(TEST_LCD_H_RES, TEST_LCD_V_RES);
    TEST_ASSERT_NOT_NULL(s_display);
    lv_display_set_flush_cb(s_display, test_flush_callback);
    lv_display_set_buffers(s_display, s_draw_buf, NULL, buffer_pixels * sizeof(lv_color_t),
                           LV_DISPLAY_RENDER_MODE_PARTIAL);

    s_flush_sem = xSemaphoreCreateBinary();
    TEST_ASSERT_NOT_NULL(s_flush_sem);
}

static void lvgl_test_env_deinit(void)
{
    lv_display_delete(s_display);
    s_display = NULL;

    free(s_draw_buf);
    s_draw_buf = NULL;

    lv_deinit();

    vSemaphoreDelete(s_flush_sem);
    s_flush_sem = NULL;
}

TEST_CASE("Validate lottie player loads built-in animation", "[lottie][lv_lottie_set_src]")
{
    lvgl_test_env_init();

    size_t lottie_size = welcome_json_end - welcome_json_start;

    /* ThorVG uses in-situ JSON parsing which modifies the data,
     * so we must copy flash data to RAM first */
    uint8_t *lottie_ram = heap_caps_malloc(lottie_size + 1, MALLOC_CAP_SPIRAM | MALLOC_CAP_8BIT);
    TEST_ASSERT_NOT_NULL(lottie_ram);
    memcpy(lottie_ram, welcome_json_start, lottie_size);
    lottie_ram[lottie_size] = '\0';  /* Null-terminate for JSON parser */

    lv_obj_t *anim = lv_lottie_create(lv_scr_act());
    TEST_ASSERT_NOT_NULL(anim);

    lv_lottie_set_size(anim, TEST_LCD_H_RES, TEST_LCD_V_RES);
    lv_lottie_set_src_data(anim, lottie_ram, lottie_size);
    TEST_ASSERT_TRUE(lv_lottie_is_loaded(anim));
    TEST_ASSERT_GREATER_THAN(0, lv_lottie_get_total_frames(anim));

    lv_refr_now(NULL);
    TEST_ASSERT_TRUE(xSemaphoreTake(s_flush_sem, pdMS_TO_TICKS(3000)));

    lv_obj_delete(anim);
    heap_caps_free(lottie_ram);
    lvgl_test_env_deinit();
}

#define TEST_MEMORY_LEAK_THRESHOLD  (700)

static size_t before_free_8bit;
static size_t before_free_32bit;

void setUp(void)
{
    before_free_8bit = heap_caps_get_free_size(MALLOC_CAP_8BIT);
    before_free_32bit = heap_caps_get_free_size(MALLOC_CAP_32BIT);
}

void tearDown(void)
{
    size_t after_free_8bit = heap_caps_get_free_size(MALLOC_CAP_8BIT);
    size_t after_free_32bit = heap_caps_get_free_size(MALLOC_CAP_32BIT);
    unity_utils_check_leak(before_free_8bit, after_free_8bit, "8BIT", TEST_MEMORY_LEAK_THRESHOLD);
    unity_utils_check_leak(before_free_32bit, after_free_32bit, "32BIT", TEST_MEMORY_LEAK_THRESHOLD);
}

void app_main(void)
{
    unity_run_menu();
}
