/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2018 Ha Thach for Adafruit Industries
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "esp_log.h"
#include "tusb.h"
#include "uf2.h"

//--------------------------------------------------------------------+
// MACRO TYPEDEF CONSTANT ENUM
//--------------------------------------------------------------------+
#define DEBUG_SPEED_TEST  0
#if DEBUG_SPEED_TEST
static uint32_t _write_ms;
#endif

static WriteState _wr_state = { 0 };
static char *TAG = "MSC";

//--------------------------------------------------------------------+
// tinyusb callbacks
//--------------------------------------------------------------------+
// Invoked when received SCSI_CMD_INQUIRY
// Application fill vendor id, product id and revision with string up to 8, 16, 4 characters respectively
void tud_msc_inquiry_cb(uint8_t lun, uint8_t vendor_id[8], uint8_t product_id[16], uint8_t product_rev[4])
{
    (void) lun;

    const char vid[] = CONFIG_TUSB_MANUFACTURER;
    const char pid[] = CONFIG_TUSB_PRODUCT;
    const char rev[] = UF2_APP_VERSION;

    memcpy(vendor_id, vid, sizeof(vid) > 8 ? 8 : sizeof(vid));
    memcpy(product_id, pid, sizeof(pid) > 16 ? 16 : sizeof(pid));
    memcpy(product_rev, rev, sizeof(rev) > 4 ? 4 : sizeof(rev));
}

// Invoked when received Test Unit Ready command.
// return true allowing host to read/write this LUN e.g SD card inserted
bool tud_msc_test_unit_ready_cb(uint8_t lun)
{
    (void) lun;
    return true;
}

// Callback invoked when received an SCSI command not in built-in list below
// - READ_CAPACITY10, READ_FORMAT_CAPACITY, INQUIRY, MODE_SENSE6, REQUEST_SENSE
// - READ10 and WRITE10 has their own callbacks
int32_t tud_msc_scsi_cb(uint8_t lun, uint8_t const scsi_cmd[16], void* buffer, uint16_t bufsize)
{
    void const* response = NULL;
    uint16_t resplen = 0;

    // most scsi handled is input
    bool in_xfer = true;

    switch (scsi_cmd[0]) {
    case SCSI_CMD_PREVENT_ALLOW_MEDIUM_REMOVAL:
        // Host is about to read/write etc ... better not to disconnect disk
        resplen = 0;
        break;

    default:
        // Set Sense = Invalid Command Operation
        tud_msc_set_sense(lun, SCSI_SENSE_ILLEGAL_REQUEST, 0x20, 0x00);

        // negative means error -> tinyusb could stall and/or response with failed status
        resplen = -1;
        break;
    }

    // return resplen must not larger than bufsize
    if (resplen > bufsize) {
        resplen = bufsize;
    }

    if (response && (resplen > 0)) {
        if (in_xfer) {
            memcpy(buffer, response, resplen);
        } else {
            // SCSI output
        }
    }

    return resplen;
}

// Callback invoked when received READ10 command.
// Copy disk's data to buffer (up to bufsize) and return number of copied bytes.
int32_t tud_msc_read10_cb(uint8_t lun, uint32_t lba, uint32_t offset, void* buffer, uint32_t bufsize)
{
    (void) lun;
    memset(buffer, 0, bufsize);

    // since we return block size each, offset should always be zero
    TU_ASSERT(offset == 0, -1);

    uint32_t count = 0;

    while (count < bufsize) {
        uf2_read_block(lba, buffer);

        lba++;
        buffer += 512;
        count  += 512;
    }

    return count;
}

// Callback invoked when received WRITE10 command.
// Process data in buffer to disk's storage and return number of written bytes
int32_t tud_msc_write10_cb(uint8_t lun, uint32_t lba, uint32_t offset, uint8_t* buffer, uint32_t bufsize)
{
    (void) lun;
    (void) offset;

    uint32_t count = 0;
    while (count < bufsize) {
        // Consider non-uf2 block write as successful
        // only break if write_block is busy with flashing (return 0)
        if (0 == uf2_write_block(lba, buffer, &_wr_state)) {
            break;
        }

        lba++;
        buffer += 512;
        count  += 512;
    }

    return count;
}

// Callback invoked when WRITE10 command is completed (status received and accepted by host).
void tud_msc_write10_complete_cb(uint8_t lun)
{
    (void) lun;
    static bool first_write = true;

    // abort the DFU, uf2 block failed integrity check
    if (_wr_state.aborted) {
        // aborted and reset
        ESP_LOGI(TAG, "STATE_WRITING_FINISHED");
    } else if (_wr_state.numBlocks) {
        // Start LED writing pattern with first write
        if (first_write) {
#if DEBUG_SPEED_TEST
            _write_ms = esp_log_timestamp();
#endif

            first_write = false;
            ESP_LOGI(TAG, "STATE_WRITING_STARTED");
        }

        // All block of uf2 file is complete --> complete DFU process
        if (_wr_state.numWritten >= _wr_state.numBlocks) {
#if DEBUG_SPEED_TEST
            uint32_t const wr_byte = _wr_state.numWritten * 256;
            _write_ms = esp_log_timestamp() - _write_ms;
            ESP_LOGI(TAG, "written %u bytes in %.02f seconds.", wr_byte, _write_ms / 1000.0F);
            ESP_LOGI(TAG, "Speed : %.02f KB/s", (wr_byte / 1000.0F) / (_write_ms / 1000.0F));
#endif

            ESP_LOGI(TAG, "STATE_WRITING_FINISHED");
            board_dfu_complete();
        }
    }
}

// Invoked when received SCSI_CMD_READ_CAPACITY_10 and SCSI_CMD_READ_FORMAT_CAPACITY to determine the disk size
// Application update block count and block size
void tud_msc_capacity_cb(uint8_t lun, uint32_t* block_count, uint16_t* block_size)
{
    (void) lun;

    *block_count = CFG_UF2_NUM_BLOCKS;
    *block_size  = CFG_UF2_BLOCK_SIZE;
}

// Invoked when received Start Stop Unit command
// - Start = 0 : stopped power mode, if load_eject = 1 : unload disk storage
// - Start = 1 : active mode, if load_eject = 1 : load disk storage
bool tud_msc_start_stop_cb(uint8_t lun, uint8_t power_condition, bool start, bool load_eject)
{
    (void) lun;
    (void) power_condition;

    if (load_eject) {
        if (start) {
            // load disk storage
        } else {
            // unload disk storage
        }
    }

    return true;
}
