/*
 * SPDX-FileCopyrightText: 2023-2025 Espressif Systems (Shanghai) CO LTD
 *
 * SPDX-License-Identifier: Apache-2.0
 */

#include <stdio.h>
#include <string.h>

#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "freertos/event_groups.h"

#include "nvs_flash.h"
#include "esp_check.h"
#include "esp_log.h"
#include "esp_event.h"
#include "esp_netif.h"

#include "driver/gpio.h"

#include "iot_eth.h"
#include "iot_eth_netif_glue.h"
#include "esp_netif_ppp.h"
#include "iot_usbh_modem.h"

static const char *TAG = "PPP_4G_DEMO";

static EventGroupHandle_t s_event_group;
#define EVENT_GOT_IP_BIT (BIT0)

// 常见 4G USB Modem VID/PID（按需要增减）
static const usb_modem_id_t usb_modem_id_list[] = {
    // SIMCom A7600C1 / A7670E
    {.match_id = {USB_DEVICE_ID_MATCH_VID_PID, 0x1E0E, 0x9011}, 5, -1, "SIMCOM, A7600C1/A7670E"},
    // SIM7080G
    {.match_id = {USB_DEVICE_ID_MATCH_VID_PID, 0x1E0E, 0x9205}, 2, -1, "SIMCOM, SIM7080G"},
    // Quectel EC20
    {.match_id = {USB_DEVICE_ID_MATCH_VID_PID, 0x2C7C, 0x0125}, 2, -1, "Quectel, EC20"},
    // Quectel EC600
    {.match_id = {USB_DEVICE_ID_MATCH_VID_PID, 0x2C7C, 0x6001}, 4, -1, "Quectel, EC600N"},
    {.match_id = {0}}, // End marker
};

static void iot_event_handle(void *arg, esp_event_base_t event_base, int32_t event_id, void *event_data)
{
    if (event_base == IOT_ETH_EVENT) {
        switch (event_id) {
        case IOT_ETH_EVENT_START:
            ESP_LOGI(TAG, "IOT_ETH_EVENT_START");
            break;
        case IOT_ETH_EVENT_STOP:
            ESP_LOGI(TAG, "IOT_ETH_EVENT_STOP");
            break;
        case IOT_ETH_EVENT_CONNECTED:
            ESP_LOGI(TAG, "IOT_ETH_EVENT_CONNECTED");
            break;
        case IOT_ETH_EVENT_DISCONNECTED:
            ESP_LOGW(TAG, "IOT_ETH_EVENT_DISCONNECTED");
            xEventGroupClearBits(s_event_group, EVENT_GOT_IP_BIT);
            break;
        default:
            ESP_LOGI(TAG, "IOT_ETH_EVENT_UNKNOWN");
            break;
        }
    } else if (event_base == IP_EVENT) {
        if (event_id == IP_EVENT_PPP_GOT_IP) {
            ESP_LOGI(TAG, "PPP GOT_IP");
            xEventGroupSetBits(s_event_group, EVENT_GOT_IP_BIT);
        } else if (event_id == IP_EVENT_PPP_LOST_IP) {
            ESP_LOGW(TAG, "PPP LOST_IP");
            xEventGroupClearBits(s_event_group, EVENT_GOT_IP_BIT);
        }
    }
}

static void usb_host_power_on_if_needed(void)
{
#ifdef CONFIG_ESP32_S3_USB_OTG
    // 这段是某些 ESP32-S3 板子的 USB HOST 供电/模式选择控制脚
    // 如果你板子不需要，可保留也不影响（只在打开 CONFIG_ESP32_S3_USB_OTG 时编译）
    const gpio_config_t io_config = {
        .pin_bit_mask = BIT64(GPIO_NUM_18),
        .mode = GPIO_MODE_OUTPUT,
        .pull_up_en = GPIO_PULLUP_DISABLE,
        .pull_down_en = GPIO_PULLDOWN_DISABLE,
        .intr_type = GPIO_INTR_DISABLE,
    };
    ESP_ERROR_CHECK(gpio_config(&io_config));
    ESP_ERROR_CHECK(gpio_set_level(GPIO_NUM_18, 1)); // select host

    const gpio_config_t power_io_config = {
        .pin_bit_mask = BIT64(GPIO_NUM_17) | BIT64(GPIO_NUM_12) | BIT64(GPIO_NUM_13),
        .mode = GPIO_MODE_OUTPUT,
        .pull_up_en = GPIO_PULLUP_DISABLE,
        .pull_down_en = GPIO_PULLDOWN_DISABLE,
        .intr_type = GPIO_INTR_DISABLE,
    };
    ESP_ERROR_CHECK(gpio_config(&power_io_config));

    ESP_ERROR_CHECK(gpio_set_level(GPIO_NUM_17, 1)); // limiter 500mA
    ESP_ERROR_CHECK(gpio_set_level(GPIO_NUM_12, 0));
    ESP_ERROR_CHECK(gpio_set_level(GPIO_NUM_13, 0)); // power off
    vTaskDelay(pdMS_TO_TICKS(10));
    ESP_ERROR_CHECK(gpio_set_level(GPIO_NUM_12, 1)); // power on
#endif
}

void app_main(void)
{
    usb_host_power_on_if_needed();

    // NVS
    esp_err_t ret = nvs_flash_init();
    if (ret == ESP_ERR_NVS_NO_FREE_PAGES || ret == ESP_ERR_NVS_NEW_VERSION_FOUND) {
        ESP_ERROR_CHECK(nvs_flash_erase());
        ESP_ERROR_CHECK(nvs_flash_init());
    }

    // netif + event loop
    ESP_ERROR_CHECK(esp_netif_init());
    ESP_ERROR_CHECK(esp_event_loop_create_default());

    s_event_group = xEventGroupCreate();
    ESP_RETURN_ON_FALSE(s_event_group != NULL, , TAG, "Failed to create event group");

    ESP_ERROR_CHECK(esp_event_handler_register(IP_EVENT, IP_EVENT_PPP_GOT_IP, iot_event_handle, NULL));
    ESP_ERROR_CHECK(esp_event_handler_register(IP_EVENT, IP_EVENT_PPP_LOST_IP, iot_event_handle, NULL));
    ESP_ERROR_CHECK(esp_event_handler_register(IOT_ETH_EVENT, ESP_EVENT_ANY_ID, iot_event_handle, NULL));

    // USBH CDC driver
    usbh_cdc_driver_config_t cdc_cfg = {
        .task_stack_size = 1024 * 4,
        .task_priority = configMAX_PRIORITIES - 1,
        .task_coreid = 0,
        .skip_init_usb_host_driver = false,
    };
    ESP_ERROR_CHECK(usbh_cdc_driver_install(&cdc_cfg));

    ESP_LOGI(TAG, "====================================");
    ESP_LOGI(TAG, "     USB 4G PPP Demo");
    ESP_LOGI(TAG, " (NO usbh_modem_start, avoid link err)");
    ESP_LOGI(TAG, "====================================");

    // Install modem (PPP)
    usbh_modem_config_t modem_config = {
        .modem_id_list = usb_modem_id_list,
        .at_tx_buffer_size = 256,
        .at_rx_buffer_size = 256,
    };

    // 关键：只用 install，不调用 usbh_modem_start()
    usbh_modem_install(&modem_config);
    ESP_LOGI(TAG, "Modem installed. Waiting PPP GOT_IP...");

    /*
    ============================================================
    Manual Dial AT Commands (UNCOMMENT & IMPLEMENT if needed)
    ============================================================
    If PPP never gets IP, usually check:
    1) USB modem inserted (power/cable)
    2) SIM inserted and ready:       AT+CPIN?
       expect: READY
    3) Network registered:           AT+CREG?
       expect: 0,1 or 0,5
    4) Signal quality:               AT+CSQ
       >10 recommended
    5) Set APN (modify to operator):
       AT+CGDCONT=1,"IP","internet"
       China Mobile : cmnet
       China Unicom : 3gnet
       China Telecom: ctnet
    6) Dial:
       ATD*99#
       or ATD*99***1#
    Notes:
    - This demo does NOT send AT by default (avoid dependency on specific AT helper APIs).
    - If you want to send AT in code, use your project's AT parser helper functions or write a small
      "send raw AT" wrapper based on the modem's AT port.
    ============================================================
    */

    // 等 PPP，带超时提示（30 秒）
    const TickType_t wait_ticks = pdMS_TO_TICKS(30000);
    EventBits_t bits = xEventGroupWaitBits(s_event_group, EVENT_GOT_IP_BIT, pdFALSE, pdFALSE, wait_ticks);

    if (!(bits & EVENT_GOT_IP_BIT)) {
        ESP_LOGW(TAG, "====================================");
        ESP_LOGW(TAG, "PPP connection TIMEOUT (30s)");
        ESP_LOGW(TAG, "Please check:");
        ESP_LOGW(TAG, "1) USB modem is plugged in (power/cable)");
        ESP_LOGW(TAG, "2) SIM is inserted and unlocked (AT+CPIN?)");
        ESP_LOGW(TAG, "3) Network registered (AT+CREG?)");
        ESP_LOGW(TAG, "4) APN is correct (AT+CGDCONT...)");
        ESP_LOGW(TAG, "Suggestion: unplug USB modem and plug again.");
        ESP_LOGW(TAG, "====================================");
    } else {
        ESP_LOGI(TAG, "PPP connected!");

        // PPP 作为默认 netif（可选）
        esp_netif_t *ppp_netif = usbh_modem_get_netif();
        if (ppp_netif) {
            esp_netif_set_default_netif(ppp_netif);
            ESP_LOGI(TAG, "PPP set as default netif");
        } else {
            ESP_LOGW(TAG, "PPP netif is NULL");
        }
    }

    // 主循环：每 10 秒提示一次当前状态（不会阻塞/不会报错）
    while (1) {
        EventBits_t now = xEventGroupGetBits(s_event_group);
        if (now & EVENT_GOT_IP_BIT) {
            ESP_LOGI(TAG, "PPP is up");
        } else {
            ESP_LOGW(TAG, "PPP is down (check USB/SIM/APN).");
        }
        vTaskDelay(pdMS_TO_TICKS(10000));
    }

    // demo 不会走到这里
    // usbh_modem_uninstall();
    // usbh_cdc_driver_uninstall();
}
