#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <pthread.h>
#include <dirent.h>
#include <time.h>

// Maximum NMEA buffer size
#define NMEA_BUF_SIZE 1024

// GPS data structure
typedef struct {
    int fix;
    int satellites;
    float hdop;
    double latitude;
    double longitude;
    float speed;
    float heading;
} gps_data_t;

gps_data_t gps_data = {0};
pthread_mutex_t gps_lock = PTHREAD_MUTEX_INITIALIZER;

volatile int running = 1;
int gps_fd = -1;

// Serial port and auto-detection functions omitted, keep original, only need open_serial_port(), find_gps_port() etc.

// convert_to_decimal_degree() and parse_gga(), parse_rmc() also remain, just remove fix_type variable in parse_gga()

double convert_to_decimal_degree(const char *val, char direction) {
    if (!val || strlen(val) < 6) return 0.0;
    double deg = 0, min = 0;
    char deg_buf[4] = {0};
    if (direction == 'N' || direction == 'S') {
        strncpy(deg_buf, val, 2);
        deg = atof(deg_buf);
        min = atof(val + 2);
    } else if (direction == 'E' || direction == 'W') {
        strncpy(deg_buf, val, 3);
        deg = atof(deg_buf);
        min = atof(val + 3);
    } else {
        return 0.0;
    }
    double dec = deg + min / 60.0;
    if (direction == 'S' || direction == 'W') dec = -dec;
    return dec;
}

int open_serial_port(const char *device, speed_t baud) {
    int fd = open(device, O_RDWR | O_NOCTTY | O_SYNC);
    if (fd < 0) {
        return -1;
    }

    struct termios tty;
    if (tcgetattr(fd, &tty) != 0) {
        close(fd);
        return -1;
    }

    cfsetospeed(&tty, baud);
    cfsetispeed(&tty, baud);

    // Configure serial port: 8N1
    tty.c_cflag = (tty.c_cflag & ~CSIZE) | CS8;
    tty.c_iflag &= ~IGNBRK;
    tty.c_lflag = 0;
    tty.c_oflag = 0;
    tty.c_cc[VMIN]  = 0;
    tty.c_cc[VTIME] = 5;

    tty.c_iflag &= ~(IXON | IXOFF | IXANY);
    tty.c_cflag |= (CLOCAL | CREAD);
    tty.c_cflag &= ~(PARENB | PARODD);
    tty.c_cflag &= ~CSTOPB;
    tty.c_cflag &= ~CRTSCTS;

    if (tcsetattr(fd, TCSANOW, &tty) != 0) {
        close(fd);
        return -1;
    }
    return fd;
}

int is_prefix(const char *str, const char *prefix) {
    return strncmp(str, prefix, strlen(prefix)) == 0;
}

int find_gps_port() {
    char buf[512];
    int fd, n;

    const char *dev_dir = "/dev";
    DIR *dir = opendir(dev_dir);
    if (!dir) {
        perror("opendir /dev");
        return -1;
    }

    struct dirent *entry;
    char path[256];

    // Try to detect ttyUSB and ttyACM devices with NMEA data
    while ((entry = readdir(dir)) != NULL) {
        if (is_prefix(entry->d_name, "ttyUSB") || is_prefix(entry->d_name, "ttyACM")) {
            snprintf(path, sizeof(path), "%s/%s", dev_dir, entry->d_name);
            printf("[Auto Detect] Trying %s\n", path);

            fd = open_serial_port(path, B460800);
            if (fd >= 0) {
                usleep(1000000);
                n = read(fd, buf, sizeof(buf) - 1);
                if (n > 0) {
                    buf[n] = '\0';
                    if (strstr(buf, "$GNGGA") || strstr(buf, "$GNRMC")) {
                        printf("[Auto Detect] GPS device found on %s\n", path);
                        closedir(dir);
                        return fd;
                    }
                }
                close(fd);
            }
        }
    }
    closedir(dir);

    printf("❌ No GPS device found automatically.\n");
    return -1;
}

void parse_gga(const char *sentence) {
    // Format: $GNGGA,time,lat,NS,lon,EW,fix,sat,hdop,...
    char *copy = strdup(sentence);
    if (!copy) return;
    char *fields[20] = {0};
    int i = 0;
    char *p = strtok(copy, ",");
    while(p && i < 20) {
        fields[i++] = p;
        p = strtok(NULL, ",");
    }
    if (i < 9) {
        free(copy);
        return;
    }
    int fix = atoi(fields[6]);
    int satellites = atoi(fields[7]);
    float hdop = atof(fields[8]);
    double latitude = convert_to_decimal_degree(fields[2], fields[3][0]);
    double longitude = convert_to_decimal_degree(fields[4], fields[5][0]);

    pthread_mutex_lock(&gps_lock);
    gps_data.fix = (fix > 0) ? 1 : 0;
    gps_data.satellites = satellites;
    gps_data.hdop = hdop;
    gps_data.latitude = latitude;
    gps_data.longitude = longitude;
    pthread_mutex_unlock(&gps_lock);

    free(copy);
}

void parse_rmc(const char *sentence) {
    // Format: $GNRMC,time,status,lat,NS,lon,EW,speed,heading,date,...
    char *copy = strdup(sentence);
    if (!copy) return;
    char *fields[20] = {0};
    int i = 0;
    char *p = strtok(copy, ",");
    while(p && i < 20) {
        fields[i++] = p;
        p = strtok(NULL, ",");
    }
    if (i < 9) {
        free(copy);
        return;
    }
    char status = fields[2][0];
    float speed = 0.0f, heading = 0.0f;
    if (status == 'A') {
        speed = atof(fields[7]);
        heading = atof(fields[8]);
    }
    pthread_mutex_lock(&gps_lock);
    gps_data.speed = speed;
    gps_data.heading = heading;
    pthread_mutex_unlock(&gps_lock);
    free(copy);
}

void print_gps_data() {
    pthread_mutex_lock(&gps_lock);
    int fix = gps_data.fix;
    int satellites = gps_data.satellites;
    float hdop = gps_data.hdop;
    double latitude = gps_data.latitude;
    double longitude = gps_data.longitude;
    float speed = gps_data.speed;
    float heading = gps_data.heading;
    pthread_mutex_unlock(&gps_lock);

    printf("-------------\n");
    printf("Positioning Status: %s\n", fix ? "Yes" : "No");
    printf("Satellites Number: %d\n", satellites);
    printf("HDOP: %.2f\n", hdop);
    printf("Latitude: %.6f\n", latitude);
    printf("Longitude: %.6f\n", longitude);
    printf("Speed (knots): %.3f\n", speed);
    printf("Heading (degrees): %.2f\n", heading);
    printf("-------------\n");
}

void *gps_thread_func(void *arg) {
    char buf[1];
    char nmea_line[NMEA_BUF_SIZE] = {0};
    int nmea_pos = 0;
    volatile int has_gga = 0;
    volatile int has_rmc = 0;

    while (running) {
        int n = read(gps_fd, buf, 1);
        if (n <= 0) {
            usleep(100000);
            continue;
        }
        char c = buf[0];
        if (c == '\n') {
            nmea_line[nmea_pos] = '\0';
            if (strncmp(nmea_line, "$GNGGA", 6) == 0) {
                parse_gga(nmea_line);
                has_gga = 1;
            } else if (strncmp(nmea_line, "$GNRMC", 6) == 0) {
                parse_rmc(nmea_line);
                has_rmc = 1;
            }
            if (has_gga && has_rmc) {
                print_gps_data();
                has_gga = 0;
                has_rmc = 0;
            }
            nmea_pos = 0;
        } else if (c != '\r') {
            if (nmea_pos < NMEA_BUF_SIZE - 1) {
                nmea_line[nmea_pos++] = c;
            } else {
                nmea_pos = 0; // Prevent overflow
            }
        }
    }
    return NULL;
}

int main() {
    gps_fd = find_gps_port();
    if (gps_fd < 0) {
        fprintf(stderr, "Exiting due to missing GPS port.\n");
        return 1;
    }

    pthread_t gps_thread;
    if (pthread_create(&gps_thread, NULL, gps_thread_func, NULL) != 0) {
        fprintf(stderr, "Failed to create GPS thread\n");
        close(gps_fd);
        return 1;
    }

    // Main thread waits, Ctrl+C to terminate
    pthread_join(gps_thread, NULL);

    close(gps_fd);
    return 0;
}