#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <pthread.h>
#include <dirent.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <time.h>
#include <errno.h>
#include <dirent.h>

// NTRIP Configuration
#define GPS_BAUD B460800
#define NTRIP_HOST "114.111.30.88"
#define NTRIP_PORT 8003
#define MOUNTPOINT "RTCM33GRCEJ"
#define NTRIP_USER "qx5946"
#define NTRIP_PASS "23832134"

// Maximum NMEA buffer size
#define NMEA_BUF_SIZE 1024

// GPS data structure
typedef struct {
    int fix_type;
    int fix;
    int satellites;
    float hdop;
    double latitude;
    double longitude;
    float speed;
    float heading;
} gps_data_t;

gps_data_t gps_data = {0};
pthread_mutex_t gps_lock = PTHREAD_MUTEX_INITIALIZER;

char last_gga[NMEA_BUF_SIZE] = {0};
time_t last_gga_time = 0;

int gps_fd = -1;
volatile int has_gga = 0;
volatile int has_rmc = 0;

volatile int running = 1;

// ----------------- Base64 Encoding -----------------
static const char base64_chars[] =
    "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
    "abcdefghijklmnopqrstuvwxyz"
    "0123456789+/";

char* base64_encode(const unsigned char *data, size_t input_length, size_t *output_length) {
    size_t i, j;
    size_t enc_len = 4 * ((input_length + 2) / 3);
    char *encoded_data = malloc(enc_len + 1);
    if (encoded_data == NULL) return NULL;

    for (i = 0, j = 0; i < input_length;) {
        uint32_t octet_a = i < input_length ? data[i++] : 0;
        uint32_t octet_b = i < input_length ? data[i++] : 0;
        uint32_t octet_c = i < input_length ? data[i++] : 0;

        uint32_t triple = (octet_a << 16) + (octet_b << 8) + octet_c;

        encoded_data[j++] = base64_chars[(triple >> 18) & 0x3F];
        encoded_data[j++] = base64_chars[(triple >> 12) & 0x3F];
        encoded_data[j++] = (i > input_length + 1) ? '=' : base64_chars[(triple >> 6) & 0x3F];
        encoded_data[j++] = (i > input_length) ? '=' : base64_chars[triple & 0x3F];
    }
    encoded_data[j] = '\0';
    if (output_length) *output_length = enc_len;
    return encoded_data;
}

// ----------------- Serial Port Configuration -----------------
int open_serial_port(const char *device, speed_t baud) {
    int fd = open(device, O_RDWR | O_NOCTTY | O_SYNC);
    if (fd < 0) {
        return -1;
    }

    struct termios tty;
    if (tcgetattr(fd, &tty) != 0) {
        close(fd);
        return -1;
    }

    cfsetospeed(&tty, baud);
    cfsetispeed(&tty, baud);

    // Configure serial port: 8N1
    tty.c_cflag = (tty.c_cflag & ~CSIZE) | CS8;     // 8 bits
    tty.c_iflag &= ~IGNBRK;                         // Disable BREAK ignore
    tty.c_lflag = 0;                                // Non-canonical mode
    tty.c_oflag = 0;                                // No output processing
    tty.c_cc[VMIN]  = 0;                            // Non-blocking read
    tty.c_cc[VTIME] = 5;                            // 0.5 second timeout

    tty.c_iflag &= ~(IXON | IXOFF | IXANY);         // Disable software flow control
    tty.c_cflag |= (CLOCAL | CREAD);                // Enable receiver, ignore modem status lines
    tty.c_cflag &= ~(PARENB | PARODD);              // No parity
    tty.c_cflag &= ~CSTOPB;                         // 1 stop bit
    tty.c_cflag &= ~CRTSCTS;                        // No hardware flow control

    if (tcsetattr(fd, TCSANOW, &tty) != 0) {
        close(fd);
        return -1;
    }
    return fd;
}

// ----------------- Auto-detect GPS Port -----------------
int is_prefix(const char *str, const char *prefix) {
    return strncmp(str, prefix, strlen(prefix)) == 0;
}

int find_gps_port() {
    char buf[512];
    int fd, n;

    const char *dev_dir = "/dev";
    DIR *dir = opendir(dev_dir);
    if (!dir) {
        perror("opendir /dev");
        return -1;
    }

    struct dirent *entry;
    char path[256];

    // First try to detect ttyUSB and ttyACM devices with NMEA data
    while ((entry = readdir(dir)) != NULL) {
        if (is_prefix(entry->d_name, "ttyUSB") || is_prefix(entry->d_name, "ttyACM")) {
            snprintf(path, sizeof(path), "%s/%s", dev_dir, entry->d_name);
            printf("[Auto Detect] Trying %s\n", path);

            fd = open_serial_port(path, GPS_BAUD);
            if (fd >= 0) {
                usleep(1000000); // Wait for serial data to stabilize
                n = read(fd, buf, sizeof(buf) - 1);
                if (n > 0) {
                    buf[n] = '\0';
                    if (strstr(buf, "$GNGGA") || strstr(buf, "$GNRMC")) {
                        printf("[Auto Detect] GPS device found on %s\n", path);
                        closedir(dir);
                        return fd;
                    }
                }
                close(fd);
            }
        }
    }
    closedir(dir);

    // Reopen directory to detect GPIO and ttyAMA devices with NMEA detection
    dir = opendir(dev_dir);
    if (!dir) {
        perror("opendir /dev");
        return -1;
    }

    // First check /dev/serial0
    snprintf(path, sizeof(path), "%s/%s", dev_dir, "serial0");
    printf("[Auto Detect GPIO] Trying %s\n", path);
    fd = open_serial_port(path, GPS_BAUD);
    if (fd >= 0) {
        usleep(1000000);
        n = read(fd, buf, sizeof(buf) - 1);
        if (n > 0) {
            buf[n] = '\0';
            if (strstr(buf, "$GNGGA") || strstr(buf, "$GNRMC")) {
                printf("[Auto Detect GPIO] GPS device found on %s\n", path);
                closedir(dir);
                return fd;
            }
        }
        close(fd);
    }

    // Then check all ttyAMA* devices
    while ((entry = readdir(dir)) != NULL) {
        if (is_prefix(entry->d_name, "ttyAMA")) {
            snprintf(path, sizeof(path), "%s/%s", dev_dir, entry->d_name);
            printf("[Auto Detect GPIO] Trying %s\n", path);

            fd = open_serial_port(path, GPS_BAUD);
            if (fd >= 0) {
                usleep(1000000);
                n = read(fd, buf, sizeof(buf) - 1);
                if (n > 0) {
                    buf[n] = '\0';
                    if (strstr(buf, "$GNGGA") || strstr(buf, "$GNRMC")) {
                        printf("[Auto Detect GPIO] GPS device found on %s\n", path);
                        closedir(dir);
                        return fd;
                    }
                }
                close(fd);
            }
        }
    }
    closedir(dir);

    printf("❌ No GPS device found automatically.\n");
    return -1;
}

// ----------------- NMEA Parsing Helper -----------------
double convert_to_decimal_degree(const char *val, char direction) {
    if (!val || strlen(val) < 6) return 0.0;
    double deg = 0, min = 0;
    char deg_buf[4] = {0};
    if (direction == 'N' || direction == 'S') {
        strncpy(deg_buf, val, 2);
        deg = atof(deg_buf);
        min = atof(val + 2);
    } else if (direction == 'E' || direction == 'W') {
        strncpy(deg_buf, val, 3);
        deg = atof(deg_buf);
        min = atof(val + 3);
    } else {
        return 0.0;
    }
    double dec = deg + min / 60.0;
    if (direction == 'S' || direction == 'W') dec = -dec;
    return dec;
}

// Parse GGA sentence
void parse_gga(const char *sentence) {
    // Format: $GNGGA,time,lat,NS,lon,EW,fix,sat,hdop,...
    // Split string
    char *copy = strdup(sentence);
    if (!copy) return;
    char *fields[20] = {0};
    int i = 0;
    char *p = strtok(copy, ",");
    while(p && i < 20) {
        fields[i++] = p;
        p = strtok(NULL, ",");
    }
    if (i < 9) {
        free(copy);
        return;
    }
    int fix_type = atoi(fields[6]);
    int satellites = atoi(fields[7]);
    float hdop = atof(fields[8]);
    double latitude = convert_to_decimal_degree(fields[2], fields[3][0]);
    double longitude = convert_to_decimal_degree(fields[4], fields[5][0]);

    pthread_mutex_lock(&gps_lock);
    gps_data.fix_type = fix_type;
    gps_data.fix = (fix_type > 0) ? 1 : 0;
    gps_data.satellites = satellites;
    gps_data.hdop = hdop;
    gps_data.latitude = latitude;
    gps_data.longitude = longitude;
    pthread_mutex_unlock(&gps_lock);

    // Save latest GGA for NTRIP
    pthread_mutex_lock(&gps_lock);
    strncpy(last_gga, sentence, sizeof(last_gga) - 3);
    strcat(last_gga, "\r\n");
    last_gga_time = time(NULL);
    pthread_mutex_unlock(&gps_lock);

    has_gga = 1;
    free(copy);
}

// Parse RMC sentence
void parse_rmc(const char *sentence) {
    // Format: $GNRMC,time,status,lat,NS,lon,EW,speed,heading,date,...
    char *copy = strdup(sentence);
    if (!copy) return;
    char *fields[20] = {0};
    int i = 0;
    char *p = strtok(copy, ",");
    while(p && i < 20) {
        fields[i++] = p;
        p = strtok(NULL, ",");
    }
    if (i < 9) {
        free(copy);
        return;
    }
    char status = fields[2][0];
    float speed = 0.0f, heading = 0.0f;
    if (status == 'A') { // Valid position
        speed = atof(fields[7]);
        heading = atof(fields[8]);
    }
    pthread_mutex_lock(&gps_lock);
    gps_data.speed = speed;
    gps_data.heading = heading;
    pthread_mutex_unlock(&gps_lock);

    has_rmc = 1;
    free(copy);
}

// Print GPS data
void print_gps_data() {
    pthread_mutex_lock(&gps_lock);
    int fix = gps_data.fix;
    int fix_type = gps_data.fix_type;
    int satellites = gps_data.satellites;
    float hdop = gps_data.hdop;
    double latitude = gps_data.latitude;
    double longitude = gps_data.longitude;
    float speed = gps_data.speed;
    float heading = gps_data.heading;
    pthread_mutex_unlock(&gps_lock);

    printf("-------------\n");
    printf("Positioning Status: %s\n", fix ? "Yes" : "No");
    printf("RTK Status: ");
    switch(fix_type) {
        case 0: printf("No Fix\n"); break;
        case 1: printf("GPS SPS Mode\n"); break;
        case 2: printf("DGNSS (SBAS/DGPS)\n"); break;
        case 3: printf("GPS PPS Fix\n"); break;
        case 4: printf("Fixed RTK\n"); break;
        case 5: printf("Float RTK\n"); break;
        default: printf("Unknown (%d)\n", fix_type); break;
    }
    printf("RTK Data: %s\n", (fix_type == 4 || fix_type == 5) ? "RECEIVED" : "NOT RECEIVED");
    printf("Satellites Number: %d\n", satellites);
    printf("HDOP: %.2f\n", hdop);
    printf("Latitude: %.6f\n", latitude);
    printf("Longitude: %.6f\n", longitude);
    printf("Speed (knots): %.3f\n", speed);
    printf("Heading (degrees): %.2f\n", heading);
    printf("-------------\n");
}

// ----------------- GPS Reading Thread -----------------
void *gps_thread_func(void *arg) {
    char buf[1];
    char nmea_line[NMEA_BUF_SIZE] = {0};
    int nmea_pos = 0;

    while (running) {
        int n = read(gps_fd, buf, 1);
        if (n <= 0) {
            usleep(100000);
            continue;
        }
        char c = buf[0];
        if (c == '\n') {
            nmea_line[nmea_pos] = '\0';
            if (strncmp(nmea_line, "$GNGGA", 6) == 0) {
                parse_gga(nmea_line);
            } else if (strncmp(nmea_line, "$GNRMC", 6) == 0) {
                parse_rmc(nmea_line);
            }
            if (has_gga && has_rmc) {
                print_gps_data();
                has_gga = 0;
                has_rmc = 0;
            }
            nmea_pos = 0;
        } else if (c != '\r') {
            if (nmea_pos < NMEA_BUF_SIZE - 1) {
                nmea_line[nmea_pos++] = c;
            } else {
                nmea_pos = 0; // Prevent overflow
            }
        }
    }
    return NULL;
}

// ----------------- NTRIP Thread -----------------
void *ntrip_thread_func(void *arg) {
    int fd = *(int*)arg;
    while (running) {
        int sockfd = socket(AF_INET, SOCK_STREAM, 0);
        if (sockfd < 0) {
            perror("socket");
            sleep(1);
            continue;
        }

        struct sockaddr_in serv_addr;
        serv_addr.sin_family = AF_INET;
        serv_addr.sin_port = htons(NTRIP_PORT);
        if (inet_pton(AF_INET, NTRIP_HOST, &serv_addr.sin_addr) <= 0) {
            fprintf(stderr, "Invalid NTRIP server IP\n");
            close(sockfd);
            sleep(1);
            continue;
        }

        if (connect(sockfd, (struct sockaddr*)&serv_addr, sizeof(serv_addr)) < 0) {
            perror("connect");
            close(sockfd);
            sleep(1);
            continue;
        }

        // Authentication string
        char auth_str[128];
        snprintf(auth_str, sizeof(auth_str), "%s:%s", NTRIP_USER, NTRIP_PASS);
        size_t auth_len = 0;
        char *auth_b64 = base64_encode((unsigned char*)auth_str, strlen(auth_str), &auth_len);
        if (!auth_b64) {
            fprintf(stderr, "base64 encode failed\n");
            close(sockfd);
            sleep(1);
            continue;
        }

        // Send request header
        char request[512];
        snprintf(request, sizeof(request),
                 "GET /%s HTTP/1.1\r\n"
                 "User-Agent: NTRIP client\r\n"
                 "Accept: */*\r\n"
                 "Connection: keep-alive\r\n"
                 "Authorization: Basic %s\r\n\r\n",
                 MOUNTPOINT, auth_b64);
        free(auth_b64);

        if (send(sockfd, request, strlen(request), 0) < 0) {
            perror("send");
            close(sockfd);
            sleep(1);
            continue;
        }
        printf("Connected to NTRIP %s:%d\n", NTRIP_HOST, NTRIP_PORT);

        // Set non-blocking timeout
        struct timeval tv;
        tv.tv_sec = 0;
        tv.tv_usec = 100000; // 100ms
        setsockopt(sockfd, SOL_SOCKET, SO_RCVTIMEO, (const char*)&tv, sizeof tv);

        time_t last_send = 0;

        while (running) {
            // Receive data and write to serial port
            char recv_buf[1024];
            ssize_t recvd = recv(sockfd, recv_buf, sizeof(recv_buf), 0);
            if (recvd > 0) {
                ssize_t written = write(fd, recv_buf, recvd);
                if (written < 0) {
                    perror("write to gps serial");
                    break;
                }
            } else if (recvd == 0) {
                printf("NTRIP server closed connection\n");
                break;
            } else if (errno != EWOULDBLOCK && errno != EAGAIN) {
                perror("recv");
                break;
            }

            // Send latest GGA every 5 seconds
            time_t now = time(NULL);
            pthread_mutex_lock(&gps_lock);
            int have_gga = (last_gga[0] != '\0');
            pthread_mutex_unlock(&gps_lock);

            if (have_gga && now - last_send >= 5) {
                pthread_mutex_lock(&gps_lock);
                ssize_t sent = send(sockfd, last_gga, strlen(last_gga), 0);
                pthread_mutex_unlock(&gps_lock);
                if (sent < 0) {
                    perror("send GGA");
                    break;
                }
                printf("[GGA sent to NTRIP] %s", last_gga);
                last_send = now;
            }
            usleep(10000); // 10ms
        }
        close(sockfd);
        printf("Reconnecting NTRIP in 1 second...\n");
        sleep(1);
    }
    return NULL;
}

// ----------------- Main Program -----------------
int main() {
    gps_fd = find_gps_port();
    if (gps_fd < 0) {
        fprintf(stderr, "Exiting due to missing GPS port.\n");
        return 1;
    }

    pthread_t gps_thread, ntrip_thread;
    if (pthread_create(&gps_thread, NULL, gps_thread_func, NULL) != 0) {
        fprintf(stderr, "Failed to create GPS thread\n");
        close(gps_fd);
        return 1;
    }
    if (pthread_create(&ntrip_thread, NULL, ntrip_thread_func, &gps_fd) != 0) {
        fprintf(stderr, "Failed to create NTRIP thread\n");
        running = 0;
        pthread_join(gps_thread, NULL);
        close(gps_fd);
        return 1;
    }

    // Main thread waits, Ctrl+C to kill
    pthread_join(gps_thread, NULL);
    running = 0;
    pthread_join(ntrip_thread, NULL);

    close(gps_fd);
    return 0;
}