import serial
import time
import threading
import glob

GPS_BAUD = 460800

gps_lock = threading.Lock()
gps_data = {
    "fix": False,
    "satellites": 0,
    "hdop": 0.0,
    "latitude": 0.0,
    "longitude": 0.0,
    "speed": 0.0,
    "heading": 0.0,
}

has_gga = False
has_rmc = False
nmea_buffer = ""

def find_gps_port(baudrate=GPS_BAUD, timeout=0.5):
    candidates = glob.glob("/dev/ttyUSB*") + glob.glob("/dev/ttyAMA*") + glob.glob("/dev/serial*") + glob.glob("/dev/ttyACM*")
    for port in candidates:
        try:
            with serial.Serial(port, baudrate=baudrate, timeout=timeout) as ser:
                time.sleep(1)
                data = ser.read(500).decode(errors="ignore")
                if "$GNGGA" in data or "$GNRMC" in data:
                    print(f"[Auto Detect] GPS device found on {port}")
                    return port
        except:
            continue

    # fallback
    gpio_port = "/dev/serial0"
    try:
        with serial.Serial(gpio_port, baudrate=baudrate, timeout=timeout) as ser:
            time.sleep(1)
            data = ser.read(500).decode(errors="ignore")
            if "$GNGGA" in data or "$GNRMC" in data:
                print(f"[Fallback] GPS device found on GPIO port {gpio_port}")
                return gpio_port
            else:
                print(f"❌ No GPS data found on fallback GPIO port {gpio_port}.")
    except Exception as e:
        print(f"❌ Could not open fallback GPIO port {gpio_port}: {e}")

    print("❌ No GPS device found automatically or on GPIO fallback.")
    return None

def convert_to_decimal_degree(val: str, direction: str) -> float:
    if len(val) < 6:
        return 0.0
    if direction in ("N", "S"):
        deg = float(val[0:2])
        minutes = float(val[2:])
    elif direction in ("E", "W"):
        deg = float(val[0:3])
        minutes = float(val[3:])
    else:
        return 0.0
    dec = deg + minutes / 60.0
    if direction in ("S", "W"):
        dec = -dec
    return dec

def split_nmea(sentence: str):
    return sentence.strip().split(',')

def parse_gga(sentence):
    global gps_data, has_gga
    parts = split_nmea(sentence)
    if len(parts) < 15:
        return
    try:
        fix_type = int(parts[6])
        satellites = int(parts[7])
        hdop = float(parts[8])
        latitude = convert_to_decimal_degree(parts[2], parts[3])
        longitude = convert_to_decimal_degree(parts[4], parts[5])
    except (ValueError, IndexError):
        return
    with gps_lock:
        gps_data.update({
            "fix": fix_type > 0,
            "satellites": satellites,
            "hdop": hdop,
            "latitude": latitude,
            "longitude": longitude
        })
    has_gga = True

def parse_rmc(sentence):
    global gps_data, has_rmc
    parts = split_nmea(sentence)
    if len(parts) < 12:
        return
    status = parts[2]
    if status == 'A':
        try:
            speed = float(parts[7])
            heading = float(parts[8])
        except ValueError:
            speed = 0.0
            heading = 0.0
    else:
        speed = 0.0
        heading = 0.0
    with gps_lock:
        gps_data.update({"speed": speed, "heading": heading})
    has_rmc = True

def print_gps_data():
    with gps_lock:
        fix = gps_data["fix"]
        satellites = gps_data["satellites"]
        hdop = gps_data["hdop"]
        latitude = gps_data["latitude"]
        longitude = gps_data["longitude"]
        speed = gps_data["speed"]
        heading = gps_data["heading"]

    print("-------------")
    print(f"Positioning Status: {'Yes' if fix else 'No'}")
    print(f"Satellites Number: {satellites}")
    print(f"HDOP: {hdop:.2f}")
    print(f"Latitude: {latitude:.6f}")
    print(f"Longitude: {longitude:.6f}")
    print(f"Speed (knots): {speed:.3f}")
    print(f"Heading (degrees): {heading:.2f}")
    print("-------------")

def gps_thread(port):
    global nmea_buffer, has_gga, has_rmc

    ser = serial.Serial(port, GPS_BAUD, timeout=0.1)

    while True:
        try:
            c = ser.read().decode(errors='ignore')
            if c == '\n':
                line = nmea_buffer.strip()
                if line.startswith("$GNGGA"):
                    parse_gga(line)
                elif line.startswith("$GNRMC"):
                    parse_rmc(line)
                nmea_buffer = ""
            elif c != '\r':
                nmea_buffer += c

            if has_gga and has_rmc:
                print_gps_data()
                has_gga = False
                has_rmc = False

        except Exception as e:
            print(f"GPS read error: {e}")
            time.sleep(0.1)

def main():
    gps_port = find_gps_port()
    if gps_port is None:
        print("Exiting due to missing GPS port.")
        return

    gps_thread(gps_port)

if __name__ == "__main__":
    main()
