/*****************************************************************************
* | File      	:   LCD_1in28_LVGL_test.c
* | Author      :   Waveshare team
* | Function    :   1.28inch LCD  test demo
* | Info        :
*----------------
* |	This version:   V1.0
* | Date        :   2024-06-27
* | Info        :
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documnetation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to  whom the Software is
# furished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS OR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
******************************************************************************/
#include "LVGL_example.h" 
#include "src/core/lv_obj.h"
#include "src/misc/lv_area.h"

// LVGL
static lv_disp_drv_t disp_drv;
static lv_disp_draw_buf_t disp_buf;
static lv_color_t buf0[DISP_HOR_RES * DISP_VER_RES];

static lv_obj_t *tv;
static lv_obj_t *tile1;
static lv_obj_t *tile2;

// AXP2101
static lv_obj_t *powerTemprature;
static lv_obj_t *isCharging;
static lv_obj_t *isDischarge;
static lv_obj_t *isStandby;
static lv_obj_t *isVbusIn;
static lv_obj_t *isVbusGood;
static lv_obj_t *chargerStatus;
static lv_obj_t *batVoltage;
static lv_obj_t *vbusVoltage;
static lv_obj_t *systemVoltage;
static lv_obj_t *batPercent;

// Touch
static uint16_t ts_x;
static uint16_t ts_y;
static uint8_t gesture = 0;
static lv_indev_state_t ts_act;
static lv_indev_drv_t indev_ts;

// Timer 
static struct repeating_timer lvgl_timer;
static struct repeating_timer axp_update_timer;

static void disp_flush_cb(lv_disp_drv_t * disp, const lv_area_t * area, lv_color_t * color_p);
static void touch_callback(uint gpio, uint32_t events);
static void ts_read_cb(lv_indev_drv_t * drv, lv_indev_data_t*data);
static void dma_handler(void);
static bool repeating_lvgl_timer_callback(struct repeating_timer *t); 
static bool repeating_axp_update_timer_callback(struct repeating_timer *t);

/********************************************************************************
function:	Initializes LVGL and enbable timers IRQ and DMA IRQ
parameter:
********************************************************************************/
void LVGL_Init(void)
{
    /*1.Init Timer*/ 
    add_repeating_timer_ms(500,  repeating_axp_update_timer_callback, NULL, &axp_update_timer);
    add_repeating_timer_ms(5,    repeating_lvgl_timer_callback,       NULL, &lvgl_timer);
    
    /*2.Init LVGL core*/
    lv_init();

    /*3.Init LVGL display*/
    lv_disp_draw_buf_init(&disp_buf, buf0, NULL, DISP_HOR_RES * DISP_VER_RES); 
    lv_disp_drv_init(&disp_drv);    
    disp_drv.flush_cb = disp_flush_cb;
    disp_drv.draw_buf = &disp_buf;        
    disp_drv.hor_res = DISP_HOR_RES;
    disp_drv.ver_res = DISP_VER_RES;
    lv_disp_t *disp= lv_disp_drv_register(&disp_drv);   

#if INPUTDEV_TS
    /*4.Init touch screen as input device*/ 
    lv_indev_drv_init(&indev_ts); 
    indev_ts.type = LV_INDEV_TYPE_POINTER;    
    indev_ts.read_cb = ts_read_cb;            
    lv_indev_t * ts_indev = lv_indev_drv_register(&indev_ts);
    //Enable touch IRQ
    DEV_KEY_Config(Touch_INT_PIN);
    DEV_IRQ_SET(Touch_INT_PIN, GPIO_IRQ_EDGE_RISE, &touch_callback);
#endif

    /*5.Init DMA for transmit color data from memory to SPI*/
    dma_channel_set_irq0_enabled(dma_tx, true);
    irq_set_exclusive_handler(DMA_IRQ_0, dma_handler);
    irq_set_enabled(DMA_IRQ_0, true);
}


/********************************************************************************
function:	Initializes the layout of LVGL widgets
parameter:
********************************************************************************/
void Widgets_Init(void)
{
    /*Style Config*/
    static lv_style_t style_roller;
    lv_style_init(&style_roller);
    lv_style_set_border_color(&style_roller, lv_palette_darken(LV_PALETTE_BLUE, 3));
    lv_style_set_text_font(&style_roller, &lv_font_montserrat_26);
    
    static lv_style_t style_imu_table;
    lv_style_init(&style_imu_table);
    lv_style_set_border_width(&style_imu_table, 1);
    lv_style_set_text_font(&style_imu_table, &lv_font_montserrat_24);

    static lv_style_t style_rtc_table;
    lv_style_init(&style_rtc_table);
    lv_style_set_border_width(&style_rtc_table, 1);
    lv_style_set_text_font(&style_rtc_table, &lv_font_montserrat_24);

    static lv_style_t style_lable;
    lv_style_init(&style_lable);
    lv_style_set_text_font(&style_lable, &lv_font_montserrat_24);

    /*Create tileview*/
    tv = lv_tileview_create(lv_scr_act());
    lv_obj_set_scrollbar_mode(tv,  LV_SCROLLBAR_MODE_OFF);

    /*Tile1: Just a pic*/
    tile1 = lv_tileview_add_tile(tv, 0, 0, LV_DIR_BOTTOM);
    
    LV_IMG_DECLARE(pic);
    lv_obj_t *img1 = lv_img_create(tile1);
    lv_img_set_src(img1, &pic);
    lv_obj_align(img1, LV_ALIGN_CENTER, 0, 0);

    /*tile2: Show AXP2101 data*/
    tile2 = lv_tileview_add_tile(tv, 0, 1, LV_DIR_TOP);

    lv_obj_t *lable1 = lv_label_create(tile2);
    lv_label_set_text(lable1, "power Temperature:");
    lv_obj_align(lable1, LV_ALIGN_LEFT_MID, 10, -150);
    lv_obj_add_style(lable1, &style_lable, 0); 
    lv_obj_t *lable2 = lv_label_create(tile2);
    lv_label_set_text(lable2, "isCharging:");
    lv_obj_align(lable2, LV_ALIGN_LEFT_MID, 10, -120);
    lv_obj_add_style(lable2, &style_lable, 0); 
    lv_obj_t *lable3 = lv_label_create(tile2);
    lv_label_set_text(lable3, "isDischarge:");
    lv_obj_align(lable3, LV_ALIGN_LEFT_MID, 10, -90);
    lv_obj_add_style(lable3, &style_lable, 0); 
    lv_obj_t *lable4 = lv_label_create(tile2);
    lv_label_set_text(lable4, "isStandby:");
    lv_obj_align(lable4, LV_ALIGN_LEFT_MID, 10, -60);
    lv_obj_add_style(lable4, &style_lable, 0); 
    lv_obj_t *lable5 = lv_label_create(tile2);
    lv_label_set_text(lable5, "isVbusin:");
    lv_obj_align(lable5, LV_ALIGN_LEFT_MID, 10, -30);
    lv_obj_add_style(lable5, &style_lable, 0); 
    lv_obj_t *lable6 = lv_label_create(tile2);
    lv_label_set_text(lable6, "isVbusGood:");
    lv_obj_align(lable6, LV_ALIGN_LEFT_MID, 10, 0);
    lv_obj_add_style(lable6, &style_lable, 0); 
    lv_obj_t *lable7 = lv_label_create(tile2);
    lv_label_set_text(lable7, "Charger Stat:");
    lv_obj_align(lable7, LV_ALIGN_LEFT_MID, 10, 30);
    lv_obj_add_style(lable7, &style_lable, 0); 
    lv_obj_t *lable8 = lv_label_create(tile2);
    lv_label_set_text(lable8, "Bettery Voltage:");
    lv_obj_align(lable8, LV_ALIGN_LEFT_MID, 10, 60);
    lv_obj_add_style(lable8, &style_lable, 0); 
    lv_obj_t *lable9 = lv_label_create(tile2);
    lv_label_set_text(lable9, "Vbus Voltage:");
    lv_obj_align(lable9, LV_ALIGN_LEFT_MID, 10, 90);
    lv_obj_add_style(lable9, &style_lable, 0); 
    lv_obj_t *lable10 = lv_label_create(tile2);
    lv_label_set_text(lable10, "System Voltage:");
    lv_obj_align(lable10, LV_ALIGN_LEFT_MID, 10, 120);
    lv_obj_add_style(lable10, &style_lable, 0); 
    lv_obj_t *lable11 = lv_label_create(tile2);
    lv_label_set_text(lable11, "Bettery Percent:");
    lv_obj_align(lable11, LV_ALIGN_LEFT_MID, 10, 150);
    lv_obj_add_style(lable11, &style_lable, 0); 

    powerTemprature = lv_label_create(tile2);
    lv_label_set_text(powerTemprature, "0");
    lv_obj_align(powerTemprature, LV_ALIGN_RIGHT_MID, -10, -150);
    lv_obj_add_style(powerTemprature, &style_lable, 0); 
    isCharging = lv_label_create(tile2);
    lv_label_set_text(isCharging, "0");
    lv_obj_align(isCharging, LV_ALIGN_RIGHT_MID, -10, -120);
    lv_obj_add_style(isCharging, &style_lable, 0); 
    isDischarge = lv_label_create(tile2);
    lv_label_set_text(isDischarge, "0");
    lv_obj_align(isDischarge, LV_ALIGN_RIGHT_MID, -10, -90);
    lv_obj_add_style(isDischarge, &style_lable, 0); 
    isStandby = lv_label_create(tile2);
    lv_label_set_text(isStandby, "0");
    lv_obj_align(isStandby, LV_ALIGN_RIGHT_MID, -10, -60);
    lv_obj_add_style(isStandby, &style_lable, 0); 
    isVbusIn = lv_label_create(tile2);
    lv_label_set_text(isVbusIn, "0");
    lv_obj_align(isVbusIn, LV_ALIGN_RIGHT_MID, -10, -30);
    lv_obj_add_style(isVbusIn, &style_lable, 0); 
    isVbusGood = lv_label_create(tile2);
    lv_label_set_text(isVbusGood, "0");
    lv_obj_align(isVbusGood, LV_ALIGN_RIGHT_MID, -10, 0);
    lv_obj_add_style(isVbusGood, &style_lable, 0); 
    chargerStatus = lv_label_create(tile2);
    lv_label_set_text(chargerStatus, "0");
    lv_obj_align(chargerStatus, LV_ALIGN_RIGHT_MID, -10, 30);
    lv_obj_add_style(chargerStatus, &style_lable, 0); 
    batVoltage = lv_label_create(tile2);
    lv_label_set_text(batVoltage, "0");
    lv_obj_align(batVoltage, LV_ALIGN_RIGHT_MID, -10, 60);
    lv_obj_add_style(batVoltage, &style_lable, 0); 
    vbusVoltage = lv_label_create(tile2);
    lv_label_set_text(vbusVoltage, "0");
    lv_obj_align(vbusVoltage, LV_ALIGN_RIGHT_MID, -10, 90);
    lv_obj_add_style(vbusVoltage, &style_lable, 0); 
    systemVoltage = lv_label_create(tile2);
    lv_label_set_text(systemVoltage, "0");
    lv_obj_align(systemVoltage, LV_ALIGN_RIGHT_MID, -10, 120);
    lv_obj_add_style(systemVoltage, &style_lable, 0); 
    batPercent = lv_label_create(tile2);
    lv_label_set_text(batPercent, "0");
    lv_obj_align(batPercent, LV_ALIGN_RIGHT_MID, -10, 150);
    lv_obj_add_style(batPercent, &style_lable, 0); 
}

/********************************************************************************
function:	Refresh image by transferring the color data to the SPI bus by DMA
parameter:
********************************************************************************/
static void disp_flush_cb(lv_disp_drv_t * disp, const lv_area_t * area, lv_color_t * color_p)
{
    // Send command in one-line mode
    QSPI_1Wrie_Mode(&qspi);
    AMOLED_1IN8_SetWindows(area->x1, area->y1, area->x2+1 , area->y2+1);  // Set the LVGL interface display position
    QSPI_Select(qspi);
    QSPI_Pixel_Write(qspi, 0x2c);

    // Four-wire mode sends RGB data
    QSPI_4Wrie_Mode(&qspi);
    channel_config_set_dreq(&c, pio_get_dreq(qspi.pio, qspi.sm, true));
    dma_channel_configure(dma_tx,   
                          &c,
                          &qspi.pio->txf[qspi.sm], 
                          color_p, // read address
                          ((area->x2 + 1 - area-> x1)*(area->y2 + 1 - area -> y1))*2,
                          true);// Start DMA transfer

    // Waiting for DMA transfer to complete
    while(dma_channel_is_busy(dma_tx));    
    QSPI_Deselect(qspi);  
}

/********************************************************************************
function:   Touch interrupt handler
parameter:
********************************************************************************/
static void touch_callback(uint gpio, uint32_t events)
{
    if (gpio == Touch_INT_PIN)
    {
        FT3168_Get_Point(); // Get coordinate data
        gesture = FT3168_Get_Gesture(); // Get gesture data
        ts_x = FT3168.x_point;
        ts_y = FT3168.y_point;
        ts_act = LV_INDEV_STATE_PRESSED;
    }
}

/********************************************************************************
function:   Update touch screen input device status
parameter:
********************************************************************************/
static void ts_read_cb(lv_indev_drv_t * drv, lv_indev_data_t*data)
{
    data->point.x = ts_x;
    data->point.y = ts_y; 
    data->state = ts_act;
    ts_act = LV_INDEV_STATE_RELEASED;
}

/********************************************************************************
function:   Indicate ready with the flushing when DMA complete transmission
parameter:
********************************************************************************/
static void dma_handler(void)
{
    if (dma_channel_get_irq0_status(dma_tx)) 
    {
        dma_channel_acknowledge_irq0(dma_tx);
        lv_disp_flush_ready(&disp_drv); // Indicate you are ready with the flushing
    }
}

/********************************************************************************
function:   Check if the page needs to be updated
parameter:
********************************************************************************/
static bool update_check(lv_obj_t *tv,lv_obj_t *tilex) 
{
    uint8_t ret = true; 

    lv_obj_t *active_tile = lv_tileview_get_tile_act(tv); // Get the current active interface
    if (active_tile != tilex) 
    {
        ret = false;
    }

    return ret;
}

/********************************************************************************
function:   Update AXP2101 data
parameter:
********************************************************************************/
static void update_axp_data()
{
    axp2101_t axp2101;
    AXP2101_ReadAll(&axp2101); //Reading AXP2101 dat1a

    char table_text[64];
    sprintf(table_text,"%4.2f*C", axp2101.powerTemprature);
    lv_label_set_text(powerTemprature, table_text);
    sprintf(table_text,"%s", axp2101.isCharging?  "YES" : "NO");
    lv_label_set_text(isCharging, table_text);
    sprintf(table_text,"%s", axp2101.isDischarge? "YES" : "NO");
    lv_label_set_text(isDischarge, table_text);
    sprintf(table_text,"%s", axp2101.isStandby?   "YES" : "NO");
    lv_label_set_text(isStandby, table_text);
    sprintf(table_text,"%s", axp2101.isVbusIn?    "YES" : "NO");
    lv_label_set_text(isVbusIn, table_text);
    sprintf(table_text,"%s", axp2101.isVbusGood?  "YES" : "NO");
    lv_label_set_text(isVbusGood, table_text);
    sprintf(table_text,"%s", axp2101.isCharging?  "YES" : "NO");

    switch (axp2101.chargerStatus) {
        case XPOWERS_AXP2101_CHG_CC_STATE:
            sprintf(table_text,"charging");
            break;
        case XPOWERS_AXP2101_CHG_CV_STATE:
            sprintf(table_text,"constant vol");
            break;
        case XPOWERS_AXP2101_CHG_DONE_STATE:
            sprintf(table_text,"charge done");
            break;
        case XPOWERS_AXP2101_CHG_STOP_STATE:
            sprintf(table_text,"not charging");
            break;
    }
    lv_label_set_text(chargerStatus, table_text);

    sprintf(table_text,"%d mv", axp2101.batVoltage);
    lv_label_set_text(batVoltage, table_text);
    sprintf(table_text,"%d mv", axp2101.vbusVoltage);
    lv_label_set_text(vbusVoltage, table_text);
    sprintf(table_text,"%d mv", axp2101.systemVoltage);
    lv_label_set_text(systemVoltage, table_text);
    sprintf(table_text,"%d %%", axp2101.batPercent);
    lv_label_set_text(batPercent, table_text);
}

/********************************************************************************
function:   Report the elapsed time to LVGL each 5ms
parameter:
********************************************************************************/
static bool repeating_lvgl_timer_callback(struct repeating_timer *t) 
{
    lv_tick_inc(5);
    return true;
}

/********************************************************************************
function:   Update AXP2101 label data each 500ms
parameter:
********************************************************************************/
static bool repeating_axp_update_timer_callback(struct repeating_timer *t) 
{
    if(update_check(tv,tile2) == true) // Need to update the interface
        update_axp_data(); // Update data

    return true;
}
