#include <cstdio>
#include <cstring>
#include <thread>
#include <chrono>

#include "RadioLib.h"
#include "hal/RPi/PiHal.h"
#include "config.h"

// 0 = LoRaWAN 1.0.x（ChirpStack 你现在用的 LoRaWAN 1.0.4，推荐）
#define USE_LW11 0

// Join 重试次数与间隔
static constexpr int JOIN_TRIES = 12;
static constexpr int JOIN_WAIT_S = 2;

// 两次 uplink 的间隔（给你时间在 ChirpStack Queue 里 Enqueue）
static constexpr int UPLINK_GAP_S = 5;

static void sleep_s(int s) {
  std::this_thread::sleep_for(std::chrono::seconds(s));
}

static uint64_t eui8_to_u64_msb(const uint8_t eui[8]) {
  uint64_t v = 0;
  for(int i = 0; i < 8; i++) v = (v << 8) | eui[i];
  return v;
}

static void printHex(const char* label, const uint8_t* buf, size_t len) {
  std::printf("%s", label);
  for(size_t i = 0; i < len; i++) {
    std::printf("%02X", buf[i]);
    if(i + 1 < len) std::printf(":");
  }
  std::printf("\n");
}

static void printDevAddr(uint32_t addr) {
  std::printf("DevAddr  = %02X%02X%02X%02X\n",
              (addr >> 24) & 0xFF, (addr >> 16) & 0xFF, (addr >> 8) & 0xFF, addr & 0xFF);
}

int main() {
  std::printf("=== LoRaWAN Diagnostic (SX1262 / EU868 / ChirpStack) ===\n");
#if USE_LW11
  std::printf("Mode: LW1.1 (beginOTAA with NwkKey + AppKey)\n");
#else
  std::printf("Mode: LW1.0.x (beginOTAA with nullptr + AppKey)\n");
#endif

  // 1) 初始化射频
  PiHal hal(LORA_SPI_BUS, LORA_SPI_HZ, 0, 0);
  SX1262 radio = new Module(&hal, LORA_CS_PIN, LORA_DIO1_PIN, LORA_RST_PIN, LORA_BUSY_PIN);

  extern const LoRaWANBand_t EU868;
  LoRaWANNode node(&radio, &EU868);

  int16_t st = radio.begin();
  if(st != RADIOLIB_ERR_NONE) {
    std::printf("❌ radio.begin failed: %d\n", st);
    return 1;
  }
  radio.setDio2AsRfSwitch(true);
  std::printf("✅ radio.begin OK\n");

  // 打印参数（确认你没填反）
  printHex("DevEUI   : ", LORAWAN_DEV_EUI, 8);
  printHex("JoinEUI  : ", LORAWAN_JOIN_EUI, 8);
  printHex("AppKey   : ", LORAWAN_APP_KEY, 16);

  uint64_t joinEUI = eui8_to_u64_msb(LORAWAN_JOIN_EUI);
  uint64_t devEUI  = eui8_to_u64_msb(LORAWAN_DEV_EUI);

  // 2) beginOTAA
#if USE_LW11
  st = node.beginOTAA(joinEUI, devEUI, LORAWAN_NWK_KEY, LORAWAN_APP_KEY);
#else
  st = node.beginOTAA(joinEUI, devEUI, nullptr, LORAWAN_APP_KEY);
#endif
  if(st != RADIOLIB_ERR_NONE) {
    std::printf("❌ beginOTAA failed: %d\n", st);
    return 2;
  }
  std::printf("✅ beginOTAA OK\n");

  // 3) Join（不打印“失败码”，只在最终失败时输出一次错误）
  std::printf("Joining...\n");
  bool joined = false;
  int16_t last_err = RADIOLIB_ERR_NONE;

  for(int i = 1; i <= JOIN_TRIES; i++) {
    st = node.activateOTAA();
    if(st == RADIOLIB_ERR_NONE) {
      joined = true;
      break;
    }
    last_err = st;
    std::printf("  Joining... (%d/%d)\n", i, JOIN_TRIES);
    sleep_s(JOIN_WAIT_S);
  }

  if(!joined) {
    std::printf("❌ OTAA Join failed, last error: %d\n", last_err);
    std::printf("Tip: if ChirpStack sees JoinRequest but no JoinAccept, check region/DR/RX1/RX2/gateway.\n");
    return 3;
  }

  std::printf("✅ OTAA Activated.\n");
  std::printf("After Join:\n");
  printDevAddr(node.getDevAddr());
  std::printf("FCntUp   = %lu\n", (unsigned long)node.getFCntUp());

  // 4) 发送两条 uplink（方便你在 ChirpStack 看 FCnt 变化；同时给你时间 enqueue 下行）
  for(int k = 1; k <= 2; k++) {
    char msg[64];
    std::snprintf(msg, sizeof(msg), "diag-2026hello-%d", k);
    std::printf("Send uplink #%d: \"%s\" (FPort=%d)\n", k, msg, UPLINK_FPORT);

    st = node.sendReceive((uint8_t*)msg, std::strlen(msg), UPLINK_FPORT, UPLINK_CONFIRMED);
    std::printf("sendReceive ret = %d\n", st);
    std::printf("FCntUp now = %lu\n", (unsigned long)node.getFCntUp());

    // 如果你在 ChirpStack -> Device -> Queue 里 Enqueue 了下行，
    // 这里的 sendReceive 可能会在 RX1/RX2 收到下行并返回相应状态。
    std::printf("Wait %ds (you can enqueue downlink in ChirpStack Queue now)...\n", UPLINK_GAP_S);
    sleep_s(UPLINK_GAP_S);
  }

  std::printf("Done.\n");
  return 0;
}
