#! /usr/bin/env python
# coding=utf-8
import os, sys
from ctypes import *
 
# ===== DLL =====
_here = os.path.dirname(os.path.abspath(__file__))
_dll  = "CH347DLLA64.DLL" if sizeof(c_void_p) == 8 else "CH347DLL.DLL"
ch347 = WinDLL(os.path.join(_here, _dll))
 
USB_ID = 0  
 
# ===== prototypes =====
ch347.CH347OpenDevice.argtypes  = [c_int]
ch347.CH347OpenDevice.restype   = c_int
ch347.CH347CloseDevice.argtypes = [c_int]
ch347.CH347CloseDevice.restype  = None
 
ch347.CH347StreamI2C.argtypes = [
    c_int, c_ulong, c_void_p,
    c_ulong, c_void_p
]
ch347.CH347StreamI2C.restype = c_int
 
def has_retack():
    return hasattr(ch347, "CH347StreamI2C_RetACK")
 
def bind_retack():
    fn = ch347.CH347StreamI2C_RetACK
    fn.argtypes = [
        c_int,
        c_ulong, c_void_p,
        c_ulong, c_void_p,
        POINTER(c_ulong)
    ]
    fn.restype = c_int
    return fn
 
def open_dev():
    if ch347.CH347OpenDevice(USB_ID) == -1:
        print("CH347 open failed")
        sys.exit(1)
 
def close_dev():
    ch347.CH347CloseDevice(USB_ID)
 
def probe_retack(fn, addr7):
    tx  = (c_ubyte * 1)()
    rx  = (c_ubyte * 1)()
    ack = c_ulong(0)
 
    tx[0] = (addr7 << 1) & 0xFE   # SLA+W
    r = fn(USB_ID, 1, tx, 0, rx, byref(ack))
    return r == 0 and ack.value > 0
 
def probe_fallback(addr7):
    tx = (c_ubyte * 1)()
    rx = (c_ubyte * 1)()
    tx[0] = ((addr7 << 1) | 1) & 0xFF
    return ch347.CH347StreamI2C(USB_ID, 1, tx, 1, rx) == 0
 
def scan():
    open_dev()
 
    use_retack = has_retack()
    fn = bind_retack() if use_retack else None
 
    print("CH347 I2C scan (Windows)")
    print("mode:", "RetACK (reliable)" if use_retack else "fallback (not reliable)")
    print("     " + " ".join(f"{i:02X}" for i in range(16)))
 
    for hi in range(0x00, 0x80, 0x10):
        row = [f"{hi:02X}:"]
        for lo in range(16):
            addr = hi + lo
            if addr > 0x7F:
                row.append("--")
                continue
 
            ok = probe_retack(fn, addr) if use_retack else probe_fallback(addr)
            row.append(f"{addr:02X}" if ok else "--")
 
        print(" ".join(row))
 
    close_dev()
 
if __name__ == "__main__":
    scan()